<?php
/**
 * Sync Manager Class
 *
 * @package SEOAuto\Plugin\Sync
 */

namespace SEOAuto\Plugin\Sync;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SEOAuto\Plugin\Api\Client;
use SEOAuto\Plugin\Publisher\ArticlePublisher;
use SEOAuto\Plugin\Support\Logger;

/**
 * Manages article synchronization with SEOAuto API.
 *
 * @since 1.0.0
 */
class SyncManager {

    /**
     * API client.
     *
     * @var Client
     */
    private Client $client;

    /**
     * Article publisher.
     *
     * @var ArticlePublisher
     */
    private ArticlePublisher $publisher;

    /**
     * Constructor.
     */
    public function __construct() {
        $this->client    = new Client();
        $this->publisher = new ArticlePublisher();
    }

    /**
     * Sync articles from SEOAuto API.
     *
     * @param bool $force_resync Force resync of all articles.
     * @return array Sync results.
     * @throws \Exception When sync fails.
     */
    public function sync( bool $force_resync = false ): array {
        Logger::info( 'Starting article sync', array( 'force' => $force_resync ) );

        $result = array(
            'new'     => 0,
            'updated' => 0,
            'failed'  => 0,
            'errors'  => array(),
        );

        try {
            // Fetch articles from API
            $articles = $this->fetch_articles( $force_resync );

            if ( empty( $articles ) ) {
                Logger::info( 'No new articles to sync' );
                return $result;
            }

            Logger::info( 'Fetched articles from API', array( 'count' => count( $articles ) ) );

            // Process each article
            foreach ( $articles as $article_data ) {
                try {
                    $is_new       = $this->is_new_article( $article_data['seoauto_id'] );
                    $publish_result = $this->publisher->publish( $article_data );

                    if ( $is_new ) {
                        $result['new']++;
                    } else {
                        $result['updated']++;
                    }

                    Logger::info(
                        'Article synced',
                        array(
                            'seoauto_id' => $article_data['seoauto_id'],
                            'post_id'     => $publish_result['post_id'],
                            'is_new'      => $is_new,
                        )
                    );

                } catch ( \Exception $e ) {
                    $result['failed']++;
                    $result['errors'][] = array(
                        'seoauto_id' => $article_data['seoauto_id'] ?? 'unknown',
                        'error'       => $e->getMessage(),
                    );

                    Logger::error(
                        'Failed to sync article',
                        array(
                            'seoauto_id' => $article_data['seoauto_id'] ?? 'unknown',
                            'error'       => $e->getMessage(),
                        )
                    );
                }
            }

        } catch ( \Exception $e ) {
            Logger::error( 'Sync failed: ' . $e->getMessage() );
            throw $e;
        }

        Logger::info( 'Sync completed', $result );

        return $result;
    }

    /**
     * Fetch articles from SEOAuto API.
     *
     * @param bool $force_resync Force resync of all articles.
     * @return array Articles data.
     * @throws \Exception When API request fails.
     */
    private function fetch_articles( bool $force_resync = false ): array {
        // STUB MODE: Return mock articles for development
        if ( defined( 'SEOAUTO_STUB_MODE' ) && SEOAUTO_STUB_MODE ) {
            return $this->get_stub_articles();
        }

        $params = array();

        if ( ! $force_resync ) {
            $last_sync = get_option( 'seoauto_last_sync', '' );

            if ( ! empty( $last_sync ) ) {
                $params['since'] = $last_sync;
            }
        }

        $response = $this->client->get( '/wp/articles/pending', $params );

        return $response['data'] ?? array();
    }

    /**
     * Check if article is new (not in local database).
     *
     * @param string $seoauto_id The SEOAuto article ID.
     * @return bool True if article is new.
     */
    private function is_new_article( string $seoauto_id ): bool {
        global $wpdb;

        $table = $wpdb->prefix . 'seoauto_articles';

        // Check if table exists first (prevents fatal errors during early plugin lifecycle).
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check.
        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) ) !== $table ) {
            return true; // Table doesn't exist, treat as new.
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Existence check.
        $exists = $wpdb->get_var(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
                "SELECT id FROM {$table} WHERE seoauto_id = %s LIMIT 1",
                $seoauto_id
            )
        );

        return empty( $exists );
    }

    /**
     * Get stub articles for development.
     *
     * @return array
     */
    private function get_stub_articles(): array {
        // Only return stub articles occasionally to simulate real behavior
        if ( wp_rand( 1, 10 ) > 3 ) {
            return array();
        }

        return array(
            array(
                'seoauto_id'        => 'art_stub_' . wp_generate_uuid4(),
                'title'              => 'Test Article: ' . gmdate( 'Y-m-d H:i:s' ),
                'content'            => '<p>This is a test article generated by SEOAuto.</p>
                <h2>Introduction</h2>
                <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.</p>
                <h2>Main Content</h2>
                <p>Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat.</p>
                <h2>Conclusion</h2>
                <p>Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur.</p>',
                'excerpt'            => 'This is a test article excerpt for development purposes.',
                'meta_description'   => 'Test meta description for SEO purposes - maximum 160 characters.',
                'focus_keyword'      => 'test article',
                'tags'               => 'test, development, stub',
                'featured_image_url' => '', // No featured image in stub mode.
                'youtube_video_id'   => '',
                'infographic_html'   => '',
            ),
        );
    }

    /**
     * Get sync statistics.
     *
     * @return array
     */
    public function get_stats(): array {
        global $wpdb;

        $table = $wpdb->prefix . 'seoauto_articles';

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $stats = $wpdb->get_results(
            "SELECT
                COUNT(*) as total,
                SUM(CASE WHEN status = 'published' THEN 1 ELSE 0 END) as published,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed
            FROM {$table}",
            ARRAY_A
        );
        // phpcs:enable

        return array(
            'total'     => (int) ( $stats[0]['total'] ?? 0 ),
            'published' => (int) ( $stats[0]['published'] ?? 0 ),
            'pending'   => (int) ( $stats[0]['pending'] ?? 0 ),
            'failed'    => (int) ( $stats[0]['failed'] ?? 0 ),
            'last_sync' => get_option( 'seoauto_last_sync', '' ),
        );
    }
}
