<?php
/**
 * Logger Class
 *
 * @package SEOAuto\Plugin\Support
 */

namespace SEOAuto\Plugin\Support;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles plugin logging to database and error_log.
 *
 * @since 1.0.0
 */
class Logger {

    /**
     * Log levels.
     */
    public const DEBUG   = 'debug';
    public const INFO    = 'info';
    public const WARNING = 'warning';
    public const ERROR   = 'error';

    /**
     * Database table name.
     *
     * @var string
     */
    private const TABLE = 'seoauto_logs';

    /**
     * Log a debug message.
     *
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     * @return void
     */
    public static function debug( string $message, array $context = array() ): void {
        if ( ! get_option( 'seoauto_debug_mode', false ) ) {
            return;
        }
        self::log( self::DEBUG, $message, $context );
    }

    /**
     * Log an info message.
     *
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     * @return void
     */
    public static function info( string $message, array $context = array() ): void {
        self::log( self::INFO, $message, $context );
    }

    /**
     * Log a warning message.
     *
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     * @return void
     */
    public static function warning( string $message, array $context = array() ): void {
        self::log( self::WARNING, $message, $context );
    }

    /**
     * Log an error message.
     *
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     * @return void
     */
    public static function error( string $message, array $context = array() ): void {
        self::log( self::ERROR, $message, $context );

        // Also log to PHP error log for errors.
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional error logging.
        error_log( '[SEOAuto] ' . $message . ' ' . wp_json_encode( $context ) );
    }

    /**
     * Log a message to the database.
     *
     * @param string $level   The log level.
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     * @return void
     */
    private static function log( string $level, string $message, array $context ): void {
        global $wpdb;

        $table = $wpdb->prefix . self::TABLE;

        // Check if table exists (might not during early activation).
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check.
        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) ) !== $table ) {
            // Fallback to error_log.
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional fallback logging.
            error_log( sprintf( '[SEOAuto][%s] %s %s', $level, $message, wp_json_encode( $context ) ) );
            return;
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Insert log entry.
        $wpdb->insert(
            $table,
            array(
                'level'      => $level,
                'message'    => $message,
                'context'    => ! empty( $context ) ? wp_json_encode( $context ) : null,
                'created_at' => current_time( 'mysql' ),
            ),
            array( '%s', '%s', '%s', '%s' )
        );
    }

    /**
     * Get recent log entries.
     *
     * @param int         $limit Maximum number of entries to return.
     * @param string|null $level Optional level filter.
     * @return array
     */
    public static function get_recent( int $limit = 100, ?string $level = null ): array {
        global $wpdb;

        $table = $wpdb->prefix . self::TABLE;

        if ( $level ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Log retrieval.
            $results = $wpdb->get_results(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
                    "SELECT * FROM {$table} WHERE level = %s ORDER BY created_at DESC LIMIT %d",
                    $level,
                    $limit
                ),
                ARRAY_A
            );
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Log retrieval.
            $results = $wpdb->get_results(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
                    "SELECT * FROM {$table} ORDER BY created_at DESC LIMIT %d",
                    $limit
                ),
                ARRAY_A
            );
        }

        return $results ?: array();
    }

    /**
     * Clear old log entries.
     *
     * @param int $days_to_keep Number of days to keep logs.
     * @return int Number of rows deleted.
     */
    public static function cleanup( int $days_to_keep = 30 ): int {
        global $wpdb;

        $table = $wpdb->prefix . self::TABLE;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup operation.
        $deleted = $wpdb->query(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
                "DELETE FROM {$table} WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
                $days_to_keep
            )
        );

        return (int) $deleted;
    }

    /**
     * Clear all log entries.
     *
     * @return int Number of rows deleted.
     */
    public static function clear_all(): int {
        global $wpdb;

        $table = $wpdb->prefix . self::TABLE;

        // Check if table exists before truncating.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check.
        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) ) !== $table ) {
            return 0;
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Truncate operation.
        $deleted = $wpdb->query( "TRUNCATE TABLE {$table}" );

        return (int) $deleted;
    }

    /**
     * Get log statistics.
     *
     * @return array
     */
    public static function get_stats(): array {
        global $wpdb;

        $table = $wpdb->prefix . self::TABLE;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Stats query.
        $stats = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
            "SELECT level, COUNT(*) as count FROM {$table} GROUP BY level",
            ARRAY_A
        );

        $result = array(
            'debug'   => 0,
            'info'    => 0,
            'warning' => 0,
            'error'   => 0,
            'total'   => 0,
        );

        foreach ( $stats as $row ) {
            $result[ $row['level'] ] = (int) $row['count'];
            $result['total']        += (int) $row['count'];
        }

        return $result;
    }
}
