<?php
/**
 * Cron Manager Class
 *
 * @package SEOAuto\Plugin\Scheduler
 */

namespace SEOAuto\Plugin\Scheduler;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SEOAuto\Plugin\Api\Client;
use SEOAuto\Plugin\Sync\SyncManager;
use SEOAuto\Plugin\Support\Logger;

/**
 * Manages WordPress cron jobs for the plugin.
 *
 * @since 1.0.0
 */
class CronManager {

    /**
     * Auto sync hook name.
     */
    public const HOOK_AUTO_SYNC = 'seoauto_auto_sync';

    /**
     * Cleanup hook name.
     */
    public const HOOK_CLEANUP = 'seoauto_cleanup';

    /**
     * Health check hook name.
     */
    public const HOOK_HEALTH_CHECK = 'seoauto_health_check';

    /**
     * Register cron callbacks.
     *
     * @return void
     */
    public function register_callbacks(): void {
        // Callbacks are registered in Plugin.php to avoid circular dependencies
    }

    /**
     * Schedule all cron events.
     *
     * @return void
     */
    public function schedule(): void {
        // Auto-sync based on settings
        $interval = get_option( 'seoauto_sync_interval', 'hourly' );
        if ( ! wp_next_scheduled( self::HOOK_AUTO_SYNC ) ) {
            wp_schedule_event( time(), $interval, self::HOOK_AUTO_SYNC );
        }

        // Daily cleanup of old logs
        if ( ! wp_next_scheduled( self::HOOK_CLEANUP ) ) {
            wp_schedule_event( time(), 'daily', self::HOOK_CLEANUP );
        }

        // Health check every 12 hours
        if ( ! wp_next_scheduled( self::HOOK_HEALTH_CHECK ) ) {
            wp_schedule_event( time(), 'twicedaily', self::HOOK_HEALTH_CHECK );
        }
    }

    /**
     * Unschedule all cron events.
     *
     * @return void
     */
    public function unschedule(): void {
        wp_clear_scheduled_hook( self::HOOK_AUTO_SYNC );
        wp_clear_scheduled_hook( self::HOOK_CLEANUP );
        wp_clear_scheduled_hook( self::HOOK_HEALTH_CHECK );
    }

    /**
     * Run auto sync job.
     *
     * @return void
     */
    public function run_auto_sync(): void {
        $api_key = get_option( 'seoauto_api_key', '' );

        if ( empty( $api_key ) ) {
            Logger::debug( 'Auto sync skipped: No API key configured' );
            return;
        }

        Logger::info( 'Starting auto sync' );

        try {
            do_action( 'seoauto_sync_started' );

            $sync_manager = new SyncManager();
            $result       = $sync_manager->sync();

            update_option( 'seoauto_last_sync', current_time( 'mysql' ) );

            do_action( 'seoauto_sync_completed', $result );

            Logger::info(
                'Auto sync completed',
                array(
                    'new'     => $result['new'] ?? 0,
                    'updated' => $result['updated'] ?? 0,
                )
            );

        } catch ( \Exception $e ) {
            Logger::error( 'Auto sync failed: ' . $e->getMessage() );
            do_action( 'seoauto_sync_failed', $e );
        }
    }

    /**
     * Run cleanup job.
     *
     * @return void
     */
    public function run_cleanup(): void {
        Logger::debug( 'Running cleanup job' );

        // Clean up old logs (keep 30 days)
        $deleted = Logger::cleanup( 30 );

        Logger::debug( 'Cleanup completed', array( 'logs_deleted' => $deleted ) );
    }

    /**
     * Run health check job.
     *
     * @return void
     */
    public function run_health_check(): void {
        $api_key = get_option( 'seoauto_api_key', '' );

        if ( empty( $api_key ) ) {
            return;
        }

        Logger::debug( 'Running health check' );

        try {
            $client = new Client();

            $client->post(
                '/wp/ping',
                array(
                    'wordpress_version' => get_bloginfo( 'version' ),
                    'plugin_version'    => SEOAUTO_VERSION,
                    'php_version'       => PHP_VERSION,
                    'site_url'          => site_url(),
                    'last_sync'         => get_option( 'seoauto_last_sync', '' ),
                    'articles_count'    => $this->get_articles_count(),
                )
            );

            Logger::debug( 'Health check sent successfully' );

        } catch ( \Exception $e ) {
            Logger::warning( 'Health check failed: ' . $e->getMessage() );
        }
    }

    /**
     * Get articles count for health check.
     *
     * @return array
     */
    private function get_articles_count(): array {
        global $wpdb;

        $table = $wpdb->prefix . 'seoauto_articles';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table.
        $counts = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
            "SELECT status, COUNT(*) as count FROM {$table} GROUP BY status",
            ARRAY_A
        );

        $result = array(
            'total'     => 0,
            'published' => 0,
            'pending'   => 0,
            'failed'    => 0,
        );

        if ( $counts ) {
            foreach ( $counts as $row ) {
                if ( isset( $result[ $row['status'] ] ) ) {
                    $result[ $row['status'] ] = (int) $row['count'];
                }
                $result['total'] += (int) $row['count'];
            }
        }

        return $result;
    }

    /**
     * Reschedule auto sync with new interval.
     *
     * @param string $interval New interval (hourly, twicedaily, daily).
     * @return void
     */
    public function reschedule_auto_sync( string $interval ): void {
        wp_clear_scheduled_hook( self::HOOK_AUTO_SYNC );

        if ( ! wp_next_scheduled( self::HOOK_AUTO_SYNC ) ) {
            wp_schedule_event( time(), $interval, self::HOOK_AUTO_SYNC );
        }
    }
}
