<?php
/**
 * Cleanup Job Class
 *
 * @package SEOAuto\Plugin\Scheduler
 */

namespace SEOAuto\Plugin\Scheduler;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SEOAuto\Plugin\Support\Logger;

/**
 * Handles database cleanup tasks.
 *
 * @since 1.0.0
 */
class CleanupJob {

    /**
     * Default number of days to keep logs.
     *
     * @var int
     */
    private const LOGS_RETENTION_DAYS = 30;

    /**
     * Run the cleanup job.
     *
     * @return array Cleanup results.
     */
    public function run(): array {
        Logger::debug( 'Starting cleanup job' );

        $results = array(
            'logs_deleted'       => 0,
            'transients_deleted' => 0,
            'orphans_cleaned'    => 0,
        );

        // Clean old logs
        $results['logs_deleted'] = $this->cleanup_old_logs();

        // Clean transients
        $results['transients_deleted'] = $this->cleanup_transients();

        // Clean orphaned records
        $results['orphans_cleaned'] = $this->cleanup_orphaned_records();

        Logger::info( 'Cleanup job completed', $results );

        return $results;
    }

    /**
     * Delete old log entries.
     *
     * @param int $days_to_keep Number of days to keep logs.
     * @return int Number of logs deleted.
     */
    private function cleanup_old_logs( int $days_to_keep = self::LOGS_RETENTION_DAYS ): int {
        global $wpdb;

        $table = $wpdb->prefix . 'seoauto_logs';

        // Check if table exists.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table existence check.
        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) ) !== $table ) {
            return 0;
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Delete old logs.
        $deleted = $wpdb->query(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
                "DELETE FROM {$table} WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
                $days_to_keep
            )
        );

        return $deleted !== false ? $deleted : 0;
    }

    /**
     * Delete expired transients.
     *
     * @return int Number of transients deleted.
     */
    private function cleanup_transients(): int {
        global $wpdb;

        // Delete expired transients.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup operation.
        $deleted = $wpdb->query(
            "DELETE a, b FROM {$wpdb->options} a, {$wpdb->options} b
            WHERE a.option_name LIKE '_transient_seoauto_%'
            AND a.option_name NOT LIKE '_transient_timeout_seoauto_%'
            AND b.option_name = CONCAT('_transient_timeout_', SUBSTRING(a.option_name, 12))
            AND b.option_value < UNIX_TIMESTAMP()"
        );

        return $deleted !== false ? $deleted : 0;
    }

    /**
     * Clean orphaned article records.
     *
     * Removes tracking records for articles whose WordPress posts no longer exist.
     *
     * @return int Number of orphaned records cleaned.
     */
    private function cleanup_orphaned_records(): int {
        global $wpdb;

        $articles_table = $wpdb->prefix . 'seoauto_articles';

        // Check if table exists.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check.
        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $articles_table ) ) !== $articles_table ) {
            return 0;
        }

        // Find orphaned records (post_id set but post doesn't exist).
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $orphaned = $wpdb->get_results(
            "SELECT a.id, a.post_id
            FROM {$articles_table} a
            LEFT JOIN {$wpdb->posts} p ON a.post_id = p.ID
            WHERE a.post_id IS NOT NULL AND p.ID IS NULL",
            ARRAY_A
        );
        // phpcs:enable

        if ( empty( $orphaned ) ) {
            return 0;
        }

        $count = 0;

        foreach ( $orphaned as $record ) {
            // Update record to show the post was deleted.
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Update operation.
            $wpdb->update(
                $articles_table,
                array(
                    'post_id' => null,
                    'status'  => 'deleted',
                ),
                array( 'id' => $record['id'] ),
                array( '%d', '%s' ),
                array( '%d' )
            );
            $count++;
        }

        return $count;
    }

    /**
     * Get cleanup statistics.
     *
     * @return array Statistics about items that could be cleaned.
     */
    public function get_stats(): array {
        global $wpdb;

        $logs_table     = $wpdb->prefix . 'seoauto_logs';
        $articles_table = $wpdb->prefix . 'seoauto_articles';

        $stats = array(
            'old_logs_count'    => 0,
            'transients_count'  => 0,
            'orphans_count'     => 0,
            'total_logs'        => 0,
            'total_articles'    => 0,
        );

        // Count old logs.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check.
        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $logs_table ) ) === $logs_table ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Stats query.
            $stats['old_logs_count'] = (int) $wpdb->get_var(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
                    "SELECT COUNT(*) FROM {$logs_table}
                    WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
                    self::LOGS_RETENTION_DAYS
                )
            );

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Stats query.
            $stats['total_logs'] = (int) $wpdb->get_var(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
                "SELECT COUNT(*) FROM {$logs_table}"
            );
        }

        // Count seoauto transients.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Stats query.
        $stats['transients_count'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->options}
            WHERE option_name LIKE '_transient_seoauto_%'"
        );

        // Count orphaned records.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check.
        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $articles_table ) ) === $articles_table ) {
            // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $stats['orphans_count'] = (int) $wpdb->get_var(
                "SELECT COUNT(*)
                FROM {$articles_table} a
                LEFT JOIN {$wpdb->posts} p ON a.post_id = p.ID
                WHERE a.post_id IS NOT NULL AND p.ID IS NULL"
            );
            // phpcs:enable

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Stats query.
            $stats['total_articles'] = (int) $wpdb->get_var(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
                "SELECT COUNT(*) FROM {$articles_table}"
            );
        }

        return $stats;
    }
}
