<?php
/**
 * Auto Sync Job Class
 *
 * @package SEOAuto\Plugin\Scheduler
 */

namespace SEOAuto\Plugin\Scheduler;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SEOAuto\Plugin\Api\Client;
use SEOAuto\Plugin\Publisher\ArticlePublisher;
use SEOAuto\Plugin\Support\Logger;

/**
 * Handles automatic synchronization with SEOAuto backend.
 *
 * @since 1.0.0
 */
class AutoSyncJob {

    /**
     * Run the auto sync job.
     *
     * @return array Sync results.
     */
    public function run(): array {
        Logger::debug( 'Starting auto sync job' );

        // Check if connected
        $api_key = get_option( 'seoauto_api_key', '' );
        $site_id = get_option( 'seoauto_site_id', '' );

        if ( empty( $api_key ) || empty( $site_id ) ) {
            Logger::debug( 'Auto sync skipped: not connected' );
            return array(
                'success' => false,
                'message' => 'Not connected to SEOAuto',
                'new'     => 0,
                'updated' => 0,
            );
        }

        // Check if auto-publish is enabled
        $auto_publish = get_option( 'seoauto_auto_publish', true );

        try {
            $client = new Client();

            // Fetch pending articles from SEOAuto
            $response = $client->get( '/wp/articles/pending' );

            if ( empty( $response['articles'] ) ) {
                Logger::debug( 'No pending articles to sync' );
                return array(
                    'success' => true,
                    'message' => 'No pending articles',
                    'new'     => 0,
                    'updated' => 0,
                );
            }

            $publisher = new ArticlePublisher();
            $new_count = 0;
            $updated_count = 0;

            foreach ( $response['articles'] as $article_data ) {
                try {
                    // Check if article already exists
                    $existing = $this->find_existing_article( $article_data['seoauto_id'] );

                    if ( $existing ) {
                        // Update existing
                        if ( $auto_publish ) {
                            $publisher->publish( $article_data );
                            $updated_count++;
                        }
                    } elseif ( $auto_publish ) {
                        // Create new and auto-publish.
                        $publisher->publish( $article_data );
                        $new_count++;
                    } else {
                        // Just save as pending.
                        $this->save_pending_article( $article_data );
                        $new_count++;
                    }

                } catch ( \Exception $e ) {
                    Logger::error(
                        'Failed to process article during sync',
                        array(
                            'seoauto_id' => $article_data['seoauto_id'] ?? '',
                            'error'       => $e->getMessage(),
                        )
                    );
                }
            }

            // Update last sync time
            update_option( 'seoauto_last_sync', current_time( 'mysql' ) );

            $result = array(
                'success' => true,
                'message' => 'Sync completed',
                'new'     => $new_count,
                'updated' => $updated_count,
            );

            Logger::info( 'Auto sync completed', $result );

            return $result;

        } catch ( \Exception $e ) {
            Logger::error( 'Auto sync failed', array( 'error' => $e->getMessage() ) );

            return array(
                'success' => false,
                'message' => $e->getMessage(),
                'new'     => 0,
                'updated' => 0,
            );
        }
    }

    /**
     * Find existing article by SEOAuto ID.
     *
     * @param string $seoauto_id The SEOAuto article ID.
     * @return array|null Article data or null.
     */
    private function find_existing_article( string $seoauto_id ): ?array {
        global $wpdb;

        $table = $wpdb->prefix . 'seoauto_articles';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table.
        return $wpdb->get_row(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, seoauto_id is prepared.
                "SELECT * FROM {$table} WHERE seoauto_id = %s",
                $seoauto_id
            ),
            ARRAY_A
        );
    }

    /**
     * Save article as pending (without publishing).
     *
     * @param array $article_data Article data from API.
     * @return void
     */
    private function save_pending_article( array $article_data ): void {
        global $wpdb;

        $table = $wpdb->prefix . 'seoauto_articles';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Insert operation.
        $wpdb->replace(
            $table,
            array(
                'seoauto_id' => $article_data['seoauto_id'],
                'title'       => $article_data['title'],
                'status'      => 'pending',
                'synced_at'   => current_time( 'mysql' ),
            ),
            array( '%s', '%s', '%s', '%s' )
        );
    }
}
