<?php
/**
 * REST Controller Base Class
 *
 * @package SEOAuto\Plugin\Rest
 */

namespace SEOAuto\Plugin\Rest;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use SEOAuto\Plugin\Api\Authentication;

/**
 * Handles REST API route registration.
 *
 * @since 1.0.0
 */
class RestController {

    /**
     * REST namespace.
     *
     * @var string
     */
    protected string $namespace = 'seoauto/v1';

    /**
     * Register all REST API routes.
     *
     * @return void
     */
    public function register_routes(): void {
        // Health check (no auth required)
        register_rest_route(
            $this->namespace,
            '/health',
            array(
                'methods'             => 'GET',
                'callback'            => array( $this, 'health_check' ),
                'permission_callback' => '__return_true',
            )
        );

        // Status endpoint (auth required)
        register_rest_route(
            $this->namespace,
            '/status',
            array(
                'methods'             => 'GET',
                'callback'            => array( $this, 'get_status' ),
                'permission_callback' => array( $this, 'check_api_key_permission' ),
            )
        );

        // Sync endpoint (auth required)
        register_rest_route(
            $this->namespace,
            '/sync',
            array(
                'methods'             => 'POST',
                'callback'            => array( $this, 'trigger_sync' ),
                'permission_callback' => array( $this, 'check_api_key_permission' ),
            )
        );

        // Articles endpoint - Create (auth required)
        register_rest_route(
            $this->namespace,
            '/articles',
            array(
                'methods'             => 'POST',
                'callback'            => array( $this, 'create_article' ),
                'permission_callback' => array( $this, 'check_api_key_permission' ),
                'args'                => $this->get_article_create_args(),
            )
        );

        // Articles endpoint - Update (auth required)
        register_rest_route(
            $this->namespace,
            '/articles/(?P<id>[\w-]+)',
            array(
                array(
                    'methods'             => 'PATCH',
                    'callback'            => array( $this, 'update_article' ),
                    'permission_callback' => array( $this, 'check_api_key_permission' ),
                    'args'                => $this->get_article_update_args(),
                ),
                array(
                    'methods'             => 'DELETE',
                    'callback'            => array( $this, 'delete_article' ),
                    'permission_callback' => array( $this, 'check_api_key_permission' ),
                ),
            )
        );

        // Articles verification endpoint - Batch verify post status (auth required)
        register_rest_route(
            $this->namespace,
            '/articles/verify',
            array(
                'methods'             => 'POST',
                'callback'            => array( $this, 'verify_articles' ),
                'permission_callback' => array( $this, 'check_api_key_permission' ),
                'args'                => array(
                    'post_ids' => array(
                        'required'          => true,
                        'type'              => 'array',
                        'description'       => __( 'Array of WordPress post IDs to verify', 'seoauto-ai-content-publisher' ),
                        'items'             => array(
                            'type' => 'integer',
                        ),
                        'sanitize_callback' => function ( $value ) {
                            return array_map( 'absint', (array) $value );
                        },
                    ),
                ),
            )
        );
    }

    /**
     * Check API key permission.
     *
     * Allows authentication via:
     * 1. Authorization Bearer header (for external API calls from SEOAuto backend)
     * 2. WordPress admin user + REST nonce (for admin dashboard calls)
     *
     * @param WP_REST_Request $request The request object.
     * @return bool|WP_Error
     */
    public function check_api_key_permission( WP_REST_Request $request ) {
        // Check rate limiting first
        $rate_limit_check = $this->check_rate_limit();
        if ( is_wp_error( $rate_limit_check ) ) {
            return $rate_limit_check;
        }

        // Method 1: Check if WordPress admin user is logged in with valid nonce
        // This allows calls from the admin dashboard without exposing API key in JS
        if ( current_user_can( 'manage_options' ) && wp_verify_nonce( $request->get_header( 'X-WP-Nonce' ), 'wp_rest' ) ) {
            return true;
        }

        // Method 2: Check Authorization Bearer header (for external API calls)
        $auth_header = $request->get_header( 'Authorization' );

        if ( empty( $auth_header ) ) {
            return new WP_Error(
                'rest_forbidden',
                __( 'Authorization header required.', 'seoauto-ai-content-publisher' ),
                array( 'status' => 401 )
            );
        }

        // Extract Bearer token
        if ( ! preg_match( '/^Bearer\s+(.+)$/i', $auth_header, $matches ) ) {
            return new WP_Error(
                'rest_forbidden',
                __( 'Invalid authorization format.', 'seoauto-ai-content-publisher' ),
                array( 'status' => 401 )
            );
        }

        $token      = $matches[1];
        $stored_key = get_option( 'seoauto_api_key', '' );

        if ( empty( $stored_key ) ) {
            return new WP_Error(
                'rest_forbidden',
                __( 'Plugin not configured.', 'seoauto-ai-content-publisher' ),
                array( 'status' => 401 )
            );
        }

        // Timing-safe comparison
        if ( ! hash_equals( $stored_key, $token ) ) {
            return new WP_Error(
                'rest_forbidden',
                __( 'Invalid API key.', 'seoauto-ai-content-publisher' ),
                array( 'status' => 403 )
            );
        }

        return true;
    }

    /**
     * Health check endpoint.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response
     */
    public function health_check( WP_REST_Request $request ): WP_REST_Response {
        global $wpdb;

        $articles_table = $wpdb->prefix . 'seoauto_articles';
        $articles_count = array();

        // Get article counts if table exists.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check.
        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $articles_table ) ) === $articles_table ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Stats query.
            $counts = $wpdb->get_results(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
                "SELECT status, COUNT(*) as count FROM {$articles_table} GROUP BY status",
                ARRAY_A
            );

            $articles_count = array(
                'total'     => 0,
                'published' => 0,
                'pending'   => 0,
                'failed'    => 0,
            );

            foreach ( $counts as $row ) {
                $articles_count[ $row['status'] ] = (int) $row['count'];
                $articles_count['total']         += (int) $row['count'];
            }
        }

        return new WP_REST_Response(
            array(
                'status'            => 'healthy',
                'wordpress_version' => get_bloginfo( 'version' ),
                'plugin_version'    => SEOAUTO_VERSION,
                'php_version'       => PHP_VERSION,
                'connected'         => ! empty( get_option( 'seoauto_api_key', '' ) ),
                'last_sync'         => get_option( 'seoauto_last_sync', null ),
                'articles_count'    => $articles_count,
            ),
            200
        );
    }

    /**
     * Get status endpoint.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response
     */
    public function get_status( WP_REST_Request $request ): WP_REST_Response {
        return new WP_REST_Response(
            array(
                'connected'       => true,
                'site_id'         => get_option( 'seoauto_site_id', '' ),
                'connected_at'    => get_option( 'seoauto_connected_at', '' ),
                'last_sync'       => get_option( 'seoauto_last_sync', '' ),
                'auto_publish'    => (bool) get_option( 'seoauto_auto_publish', true ),
                'plugin_version'  => SEOAUTO_VERSION,
                'wordpress_version' => get_bloginfo( 'version' ),
            ),
            200
        );
    }

    /**
     * Trigger sync endpoint.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error
     */
    public function trigger_sync( WP_REST_Request $request ) {
        try {
            do_action( 'seoauto_sync_started' );

            $sync_manager = new \SEOAuto\Plugin\Sync\SyncManager();
            $result       = $sync_manager->sync();

            update_option( 'seoauto_last_sync', current_time( 'mysql' ) );

            do_action( 'seoauto_sync_completed', $result );

            return new WP_REST_Response(
                array(
                    'success'         => true,
                    'message'         => __( 'Sync completed successfully.', 'seoauto-ai-content-publisher' ),
                    'articles_new'    => $result['new'] ?? 0,
                    'articles_updated' => $result['updated'] ?? 0,
                ),
                200
            );

        } catch ( \Exception $e ) {
            do_action( 'seoauto_sync_failed', $e );

            return new WP_Error(
                'sync_failed',
                $e->getMessage(),
                array( 'status' => 500 )
            );
        }
    }

    /**
     * Create article endpoint.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error
     */
    public function create_article( WP_REST_Request $request ) {
        $params = $request->get_json_params();

        do_action( 'seoauto_article_received', $params );

        try {
            $publisher = new \SEOAuto\Plugin\Publisher\ArticlePublisher();
            $result    = $publisher->publish( $params );

            do_action( 'seoauto_article_published', $result['post_id'], $params );

            return new WP_REST_Response(
                array(
                    'success'   => true,
                    'post_id'   => $result['post_id'],
                    'permalink' => $result['permalink'],
                    'message'   => __( 'Article published successfully.', 'seoauto-ai-content-publisher' ),
                ),
                201
            );

        } catch ( \Exception $e ) {
            do_action( 'seoauto_article_failed', $params, $e );

            return new WP_Error(
                'publish_failed',
                $e->getMessage(),
                array( 'status' => 500 )
            );
        }
    }

    /**
     * Update article endpoint.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error
     */
    public function update_article( WP_REST_Request $request ) {
        $seoauto_id = $request->get_param( 'id' );
        $params      = $request->get_json_params();

        $params['seoauto_id'] = $seoauto_id;

        try {
            $publisher = new \SEOAuto\Plugin\Publisher\ArticlePublisher();
            $result    = $publisher->publish( $params );

            do_action( 'seoauto_article_updated', $result['post_id'], $params );

            return new WP_REST_Response(
                array(
                    'success'   => true,
                    'post_id'   => $result['post_id'],
                    'permalink' => $result['permalink'],
                    'message'   => __( 'Article updated successfully.', 'seoauto-ai-content-publisher' ),
                ),
                200
            );

        } catch ( \Exception $e ) {
            return new WP_Error(
                'update_failed',
                $e->getMessage(),
                array( 'status' => 500 )
            );
        }
    }

    /**
     * Delete article endpoint.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error
     */
    public function delete_article( WP_REST_Request $request ) {
        global $wpdb;

        $seoauto_id    = $request->get_param( 'id' );
        $articles_table = $wpdb->prefix . 'seoauto_articles';

        // Find the article.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table.
        $article = $wpdb->get_row(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, seoauto_id is prepared.
                "SELECT * FROM {$articles_table} WHERE seoauto_id = %s",
                $seoauto_id
            ),
            ARRAY_A
        );

        if ( ! $article ) {
            return new WP_Error(
                'not_found',
                __( 'Article not found.', 'seoauto-ai-content-publisher' ),
                array( 'status' => 404 )
            );
        }

        // Trash the WordPress post if exists
        if ( ! empty( $article['post_id'] ) ) {
            wp_trash_post( $article['post_id'] );
        }

        // Update status in tracking table.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Update operation.
        $wpdb->update(
            $articles_table,
            array( 'status' => 'deleted' ),
            array( 'seoauto_id' => $seoauto_id ),
            array( '%s' ),
            array( '%s' )
        );

        do_action( 'seoauto_article_deleted', $article['post_id'] ?? null );

        return new WP_REST_Response(
            array(
                'success' => true,
                'message' => __( 'Article deleted successfully.', 'seoauto-ai-content-publisher' ),
            ),
            200
        );
    }

    /**
     * Verify articles endpoint.
     * Checks if WordPress posts exist and returns their status.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response
     */
    public function verify_articles( WP_REST_Request $request ): WP_REST_Response {
        $post_ids = $request->get_param( 'post_ids' );
        $results  = array();

        foreach ( $post_ids as $post_id ) {
            $post_id = absint( $post_id );

            if ( 0 === $post_id ) {
                $results[ (string) $post_id ] = 'invalid';
                continue;
            }

            $post = get_post( $post_id );

            if ( ! $post ) {
                // Post doesn't exist at all - permanently deleted
                $results[ (string) $post_id ] = 'deleted';
            } elseif ( 'trash' === $post->post_status ) {
                // Post is in trash
                $results[ (string) $post_id ] = 'trashed';
            } elseif ( in_array( $post->post_status, array( 'publish', 'draft', 'pending', 'private', 'future' ), true ) ) {
                // Post exists with a valid status
                $results[ (string) $post_id ] = 'live';
            } else {
                // Unknown status
                $results[ (string) $post_id ] = 'unknown';
            }
        }

        return new WP_REST_Response(
            array(
                'results'     => $results,
                'verified_at' => current_time( 'c' ),
            ),
            200
        );
    }

    /**
     * Get article creation arguments.
     *
     * @return array
     */
    protected function get_article_create_args(): array {
        return array(
            'seoauto_id'        => array(
                'required'          => true,
                'type'              => 'string',
                'description'       => __( 'Unique article ID from SEOAuto', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'title'              => array(
                'required'          => true,
                'type'              => 'string',
                'description'       => __( 'Article title', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'content'            => array(
                'required'          => true,
                'type'              => 'string',
                'description'       => __( 'Article HTML content', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'wp_kses_post',
            ),
            'excerpt'            => array(
                'type'              => 'string',
                'description'       => __( 'Article excerpt', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'sanitize_textarea_field',
            ),
            'meta_description'   => array(
                'type'              => 'string',
                'description'       => __( 'SEO meta description', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'meta_keywords'      => array(
                'type'              => 'string',
                'description'       => __( 'SEO keywords', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'focus_keyword'      => array(
                'type'              => 'string',
                'description'       => __( 'Primary focus keyword', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'tags'               => array(
                'type'              => 'string',
                'description'       => __( 'Comma-separated tags', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'featured_image_url' => array(
                'type'              => 'string',
                'format'            => 'uri',
                'description'       => __( 'Featured image URL', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'esc_url_raw',
            ),
            'infographic_html'   => array(
                'type'              => 'string',
                'description'       => __( 'Infographic HTML content', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'wp_kses_post',
            ),
            'youtube_video_id'   => array(
                'type'              => 'string',
                'description'       => __( 'YouTube video ID to embed', 'seoauto-ai-content-publisher' ),
                'sanitize_callback' => 'sanitize_text_field',
            ),
        );
    }

    /**
     * Get article update arguments.
     *
     * @return array
     */
    protected function get_article_update_args(): array {
        $args = $this->get_article_create_args();

        // Make all fields optional for updates
        foreach ( $args as $key => $arg ) {
            $args[ $key ]['required'] = false;
        }

        return $args;
    }

    /**
     * Check rate limiting for API requests.
     *
     * Limits requests to 100 per minute per IP address to prevent abuse/DoS.
     *
     * @return bool|WP_Error True if allowed, WP_Error if rate limited.
     */
    private function check_rate_limit() {
        // Get client IP (sanitized)
        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '0.0.0.0';

        // Sanitize IP for use in transient key
        $ip_key = preg_replace( '/[^a-zA-Z0-9]/', '', $ip );
        $transient_key = 'seoauto_rate_' . substr( md5( $ip_key ), 0, 12 );

        // Get current request count
        $requests = (int) get_transient( $transient_key );

        // Check if over limit (100 requests per minute)
        $limit = apply_filters( 'seoauto_rate_limit', 100 );

        if ( $requests >= $limit ) {
            return new WP_Error(
                'rate_limit_exceeded',
                __( 'Too many requests. Please wait before trying again.', 'seoauto-ai-content-publisher' ),
                array( 'status' => 429 )
            );
        }

        // Increment counter
        set_transient( $transient_key, $requests + 1, 60 );

        return true;
    }
}
