<?php
/**
 * SEO Integration Class
 *
 * @package SEOAuto\Plugin\Publisher
 */

namespace SEOAuto\Plugin\Publisher;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles integration with SEO plugins (Yoast, Rank Math, AIOSEO, SEOPress).
 *
 * @since 1.0.0
 */
class SeoIntegration {

    /**
     * Set SEO metadata based on active SEO plugin.
     *
     * @param int   $post_id The post ID.
     * @param array $meta    SEO metadata array.
     * @return void
     */
    public function set_metadata( int $post_id, array $meta ): void {
        /**
         * Filter the SEO metadata before saving.
         *
         * @param array $meta    The SEO metadata.
         * @param int   $post_id The post ID.
         */
        $meta = apply_filters( 'seoauto_seo_metadata', $meta, $post_id );

        // Yoast SEO
        if ( $this->is_yoast_active() ) {
            $this->set_yoast_meta( $post_id, $meta );
        }

        // Rank Math
        if ( $this->is_rank_math_active() ) {
            $this->set_rank_math_meta( $post_id, $meta );
        }

        // All in One SEO
        if ( $this->is_aioseo_active() ) {
            $this->set_aioseo_meta( $post_id, $meta );
        }

        // SEOPress
        if ( $this->is_seopress_active() ) {
            $this->set_seopress_meta( $post_id, $meta );
        }

        // Always store as native meta (fallback)
        if ( ! empty( $meta['meta_description'] ) ) {
            update_post_meta( $post_id, '_seoauto_meta_description', $meta['meta_description'] );
        }

        if ( ! empty( $meta['focus_keyword'] ) ) {
            update_post_meta( $post_id, '_seoauto_focus_keyword', $meta['focus_keyword'] );
        }

        if ( ! empty( $meta['meta_keywords'] ) ) {
            update_post_meta( $post_id, '_seoauto_meta_keywords', $meta['meta_keywords'] );
        }
    }

    /**
     * Yoast SEO integration.
     *
     * @param int   $post_id The post ID.
     * @param array $meta    SEO metadata.
     * @return void
     */
    private function set_yoast_meta( int $post_id, array $meta ): void {
        if ( ! empty( $meta['meta_description'] ) ) {
            update_post_meta( $post_id, '_yoast_wpseo_metadesc', $meta['meta_description'] );
        }

        if ( ! empty( $meta['focus_keyword'] ) ) {
            update_post_meta( $post_id, '_yoast_wpseo_focuskw', $meta['focus_keyword'] );
        }
    }

    /**
     * Rank Math integration.
     *
     * @param int   $post_id The post ID.
     * @param array $meta    SEO metadata.
     * @return void
     */
    private function set_rank_math_meta( int $post_id, array $meta ): void {
        if ( ! empty( $meta['meta_description'] ) ) {
            update_post_meta( $post_id, 'rank_math_description', $meta['meta_description'] );
        }

        if ( ! empty( $meta['focus_keyword'] ) ) {
            update_post_meta( $post_id, 'rank_math_focus_keyword', $meta['focus_keyword'] );
        }
    }

    /**
     * All in One SEO integration.
     *
     * @param int   $post_id The post ID.
     * @param array $meta    SEO metadata.
     * @return void
     */
    private function set_aioseo_meta( int $post_id, array $meta ): void {
        if ( ! empty( $meta['meta_description'] ) ) {
            update_post_meta( $post_id, '_aioseo_description', $meta['meta_description'] );
        }

        if ( ! empty( $meta['focus_keyword'] ) ) {
            // AIOSEO stores keyphrases as JSON
            $keyphrases = array(
                'focus' => array(
                    'keyphrase' => $meta['focus_keyword'],
                ),
            );
            update_post_meta( $post_id, '_aioseo_keyphrases', wp_json_encode( $keyphrases ) );
        }
    }

    /**
     * SEOPress integration.
     *
     * @param int   $post_id The post ID.
     * @param array $meta    SEO metadata.
     * @return void
     */
    private function set_seopress_meta( int $post_id, array $meta ): void {
        if ( ! empty( $meta['meta_description'] ) ) {
            update_post_meta( $post_id, '_seopress_titles_desc', $meta['meta_description'] );
        }

        if ( ! empty( $meta['focus_keyword'] ) ) {
            update_post_meta( $post_id, '_seopress_analysis_target_kw', $meta['focus_keyword'] );
        }
    }

    /**
     * Check if Yoast SEO is active.
     *
     * @return bool
     */
    public function is_yoast_active(): bool {
        return defined( 'WPSEO_VERSION' );
    }

    /**
     * Check if Rank Math is active.
     *
     * @return bool
     */
    public function is_rank_math_active(): bool {
        return class_exists( 'RankMath' );
    }

    /**
     * Check if All in One SEO is active.
     *
     * @return bool
     */
    public function is_aioseo_active(): bool {
        return defined( 'AIOSEO_VERSION' );
    }

    /**
     * Check if SEOPress is active.
     *
     * @return bool
     */
    public function is_seopress_active(): bool {
        return defined( 'SEOPRESS_VERSION' );
    }

    /**
     * Get the active SEO plugin name.
     *
     * @return string|null Plugin name or null if none active.
     */
    public function get_active_seo_plugin(): ?string {
        if ( $this->is_yoast_active() ) {
            return 'Yoast SEO';
        }

        if ( $this->is_rank_math_active() ) {
            return 'Rank Math';
        }

        if ( $this->is_aioseo_active() ) {
            return 'All in One SEO';
        }

        if ( $this->is_seopress_active() ) {
            return 'SEOPress';
        }

        return null;
    }

    /**
     * Check if any SEO plugin is active.
     *
     * @return bool
     */
    public function has_seo_plugin(): bool {
        return $this->is_yoast_active()
            || $this->is_rank_math_active()
            || $this->is_aioseo_active()
            || $this->is_seopress_active();
    }
}
