<?php
/**
 * Post Event Handler
 *
 * Handles WordPress post events (delete, trash, restore) and notifies SEOAuto backend.
 *
 * @package SEOAuto\Plugin\Publisher
 */

namespace SEOAuto\Plugin\Publisher;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SEOAuto\Plugin\Api\Client;
use SEOAuto\Plugin\Support\Logger;

/**
 * Handles post events and syncs status with SEOAuto backend.
 *
 * @since 1.2.0
 */
class PostEventHandler {

	/**
	 * API client instance.
	 *
	 * @var Client
	 */
	private Client $client;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->client = new Client();
	}

	/**
	 * Register WordPress hooks for post events.
	 *
	 * @return void
	 */
	public function register_hooks(): void {
		// Before a post is permanently deleted.
		add_action( 'before_delete_post', array( $this, 'handle_post_deleted' ), 10, 2 );

		// When a post is moved to trash.
		add_action( 'wp_trash_post', array( $this, 'handle_post_trashed' ) );

		// When a post is restored from trash.
		add_action( 'untrashed_post', array( $this, 'handle_post_untrashed' ) );
	}

	/**
	 * Handle post permanent deletion.
	 *
	 * @param int      $post_id The post ID being deleted.
	 * @param \WP_Post $post    The post object.
	 * @return void
	 */
	public function handle_post_deleted( int $post_id, \WP_Post $post ): void {
		// Only handle posts.
		if ( 'post' !== $post->post_type ) {
			return;
		}

		$seoauto_id = $this->get_seoauto_article_id( $post_id );

		if ( ! $seoauto_id ) {
			return;
		}

		Logger::info(
			'Post deleted, notifying SEOAuto',
			array(
				'post_id'    => $post_id,
				'seoauto_id' => $seoauto_id,
			)
		);

		$this->notify_backend( $seoauto_id, 'deleted' );
	}

	/**
	 * Handle post moved to trash.
	 *
	 * @param int $post_id The post ID being trashed.
	 * @return void
	 */
	public function handle_post_trashed( int $post_id ): void {
		$post = get_post( $post_id );

		// Only handle posts.
		if ( ! $post || 'post' !== $post->post_type ) {
			return;
		}

		$seoauto_id = $this->get_seoauto_article_id( $post_id );

		if ( ! $seoauto_id ) {
			return;
		}

		Logger::info(
			'Post trashed, notifying SEOAuto',
			array(
				'post_id'    => $post_id,
				'seoauto_id' => $seoauto_id,
			)
		);

		$this->notify_backend( $seoauto_id, 'trashed' );
	}

	/**
	 * Handle post restored from trash.
	 *
	 * @param int $post_id The post ID being restored.
	 * @return void
	 */
	public function handle_post_untrashed( int $post_id ): void {
		$post = get_post( $post_id );

		// Only handle posts.
		if ( ! $post || 'post' !== $post->post_type ) {
			return;
		}

		$seoauto_id = $this->get_seoauto_article_id( $post_id );

		if ( ! $seoauto_id ) {
			return;
		}

		Logger::info(
			'Post restored, notifying SEOAuto',
			array(
				'post_id'    => $post_id,
				'seoauto_id' => $seoauto_id,
			)
		);

		$this->notify_backend( $seoauto_id, 'restored' );
	}

	/**
	 * Get the SEOAuto article ID for a WordPress post.
	 *
	 * @param int $post_id WordPress post ID.
	 * @return string|null SEOAuto article ID or null if not found.
	 */
	private function get_seoauto_article_id( int $post_id ): ?string {
		// Check post meta first.
		$seoauto_id = get_post_meta( $post_id, '_seoauto_article_id', true );

		if ( ! empty( $seoauto_id ) ) {
			return $seoauto_id;
		}

		// Fall back to database table.
		global $wpdb;

		$articles_table = $wpdb->prefix . 'seoauto_articles';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Lookup query.
		$seoauto_id = $wpdb->get_var(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
				"SELECT seoauto_id FROM {$articles_table} WHERE post_id = %d",
				$post_id
			)
		);

		return $seoauto_id ? $seoauto_id : null;
	}

	/**
	 * Notify SEOAuto backend about post status change.
	 *
	 * @param string $seoauto_id The SEOAuto article ID.
	 * @param string $event      The event type: 'deleted', 'trashed', or 'restored'.
	 * @return bool True on success, false on failure.
	 */
	private function notify_backend( string $seoauto_id, string $event ): bool {
		// Check if plugin is connected.
		$api_key = get_option( 'seoauto_api_key', '' );
		if ( empty( $api_key ) ) {
			Logger::debug(
				'Skipping notification - plugin not connected',
				array(
					'seoauto_id' => $seoauto_id,
					'event'      => $event,
				)
			);
			return false;
		}

		try {
			$endpoint = sprintf( 'wp/articles/%s/%s', $seoauto_id, $event );

			$this->client->post( $endpoint );

			Logger::info(
				'Successfully notified SEOAuto',
				array(
					'seoauto_id' => $seoauto_id,
					'event'      => $event,
				)
			);

			return true;

		} catch ( \Exception $e ) {
			Logger::error(
				'Failed to notify SEOAuto',
				array(
					'seoauto_id' => $seoauto_id,
					'event'      => $event,
					'error'      => $e->getMessage(),
				)
			);

			return false;
		}
	}
}
