<?php
/**
 * Image Handler Class
 *
 * @package SEOAuto\Plugin\Publisher
 */

namespace SEOAuto\Plugin\Publisher;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use WP_Error;
use SEOAuto\Plugin\Support\Logger;

/**
 * Handles image downloading and attachment to posts.
 *
 * @since 1.0.0
 */
class ImageHandler {

    /**
     * Download image from URL and attach to post.
     *
     * @param string $url     Remote image URL.
     * @param int    $post_id WordPress post ID.
     * @param string $title   Image title.
     * @return int|WP_Error Attachment ID or error.
     */
    public function download_and_attach( string $url, int $post_id, string $title = '' ): int|WP_Error {
        // Require WordPress media functions
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        /**
         * Filter the featured image URL before downloading.
         *
         * @param string $url   The image URL.
         * @param array  $data  The article data.
         */
        $url = apply_filters( 'seoauto_featured_image_url', $url, array() );

        // Validate URL
        if ( ! filter_var( $url, FILTER_VALIDATE_URL ) ) {
            return new WP_Error( 'invalid_url', __( 'Invalid image URL', 'seoauto-ai-content-publisher' ) );
        }

        Logger::debug( 'Downloading image', array( 'url' => $url ) );

        // Download to temp file
        $tmp_file = download_url( $url, 60 );

        if ( is_wp_error( $tmp_file ) ) {
            Logger::warning( 'Failed to download image', array( 'error' => $tmp_file->get_error_message() ) );
            return $tmp_file;
        }

        // Validate MIME type of downloaded file (security: prevent uploading PHP files with image extensions)
        $allowed_mime_types = array( 'image/jpeg', 'image/png', 'image/gif', 'image/webp' );
        $file_info          = wp_check_filetype_and_ext( $tmp_file, basename( $url ) );
        $detected_type      = $file_info['type'];

        // If wp_check_filetype_and_ext couldn't determine type, try getimagesize
        if ( empty( $detected_type ) && function_exists( 'getimagesize' ) ) {
            $image_info    = @getimagesize( $tmp_file );
            $detected_type = $image_info['mime'] ?? '';
        }

        if ( empty( $detected_type ) || ! in_array( $detected_type, $allowed_mime_types, true ) ) {
            wp_delete_file( $tmp_file );
            Logger::warning( 'Invalid image MIME type', array( 'detected' => $detected_type, 'url' => $url ) );
            return new WP_Error( 'invalid_mime_type', __( 'Invalid image type detected', 'seoauto-ai-content-publisher' ) );
        }

        // Get file extension from URL or content type
        $extension = $this->get_extension_from_url( $url );

        /**
         * Filter the image title before saving.
         *
         * @param string $title The image title.
         * @param array  $data  The article data.
         */
        $title = apply_filters( 'seoauto_image_title', $title, array() );

        // Generate clean filename
        $filename = sanitize_file_name( $title ?: 'seoauto-image-' . time() ) . '.' . $extension;

        // Prepare file array for sideload
        $file_array = array(
            'name'     => $filename,
            'tmp_name' => $tmp_file,
        );

        // Sideload into media library
        $attachment_id = media_handle_sideload( $file_array, $post_id, $title );

        // Clean up temp file if sideload failed
        if ( is_wp_error( $attachment_id ) ) {
            wp_delete_file( $tmp_file );
            Logger::warning( 'Failed to sideload image', array( 'error' => $attachment_id->get_error_message() ) );
            return $attachment_id;
        }

        // Set alt text
        update_post_meta( $attachment_id, '_wp_attachment_image_alt', $title );

        Logger::debug(
            'Image downloaded and attached',
            array(
                'attachment_id' => $attachment_id,
                'post_id'       => $post_id,
            )
        );

        return $attachment_id;
    }

    /**
     * Extract file extension from URL.
     *
     * @param string $url The image URL.
     * @return string File extension.
     */
    private function get_extension_from_url( string $url ): string {
        $path      = wp_parse_url( $url, PHP_URL_PATH );
        $extension = pathinfo( $path, PATHINFO_EXTENSION );

        // Remove query string if present
        $extension = strtok( $extension, '?' );

        // Validate extension
        $allowed = array( 'jpg', 'jpeg', 'png', 'gif', 'webp' );

        if ( empty( $extension ) || ! in_array( strtolower( $extension ), $allowed, true ) ) {
            return 'jpg'; // Default to jpg
        }

        return strtolower( $extension );
    }

    /**
     * Delete attachment by ID.
     *
     * @param int  $attachment_id The attachment ID.
     * @param bool $force_delete  Whether to force delete.
     * @return bool True on success.
     */
    public function delete_attachment( int $attachment_id, bool $force_delete = false ): bool {
        $result = wp_delete_attachment( $attachment_id, $force_delete );
        return false !== $result;
    }
}
