<?php
/**
 * Content Processor Class
 *
 * @package SEOAuto\Plugin\Publisher
 */

namespace SEOAuto\Plugin\Publisher;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Processes article content before publishing.
 *
 * @since 1.0.0
 */
class ContentProcessor {

    /**
     * Process article content.
     *
     * @param string $content The raw content.
     * @param array  $options Processing options.
     * @return string Processed content.
     */
    public function process( string $content, array $options = array() ): string {
        /**
         * Filter the content before processing.
         *
         * @param string $content The raw content.
         * @param array  $options The processing options.
         */
        $content = apply_filters( 'seoauto_before_publish_content', $content, $options );

        // Infographic is already embedded in content at the end by backend.
        // The inject_infographic method is kept for backwards compatibility but not called.
        // Backend controls infographic placement, plugin should not inject separately.

        // YouTube video is already embedded in content by backend at 75% position.
        // The inject_youtube method is kept for backwards compatibility but not called.
        // Backend controls video placement, plugin should not inject separately.

        /**
         * Filter the content after processing.
         *
         * @param string $content The processed content.
         * @param array  $options The processing options.
         */
        $content = apply_filters( 'seoauto_after_process_content', $content, $options );

        return $content;
    }

    /**
     * Inject infographic into content.
     *
     * @param string $content        The content.
     * @param string $infographic_html The infographic HTML.
     * @return string Content with infographic.
     */
    private function inject_infographic( string $content, string $infographic_html ): string {
        $position = get_option( 'seoauto_infographic_position', 'end' );

        // Wrap infographic in container (sanitize HTML to prevent XSS)
        $infographic = '<div class="seoauto-infographic">' . wp_kses_post( $infographic_html ) . '</div>';

        if ( 'after_first_p' === $position ) {
            // Find first closing </p> tag and inject after it
            $pos = strpos( $content, '</p>' );

            if ( false !== $pos ) {
                $content = substr_replace( $content, '</p>' . $infographic, $pos, 4 );
            } else {
                // No paragraph found, prepend
                $content = $infographic . $content;
            }
        } else {
            // Append to end
            $content .= $infographic;
        }

        return $content;
    }

    /**
     * Inject YouTube video into content.
     *
     * @param string $content  The content.
     * @param string $video_id The YouTube video ID.
     * @return string Content with YouTube embed.
     */
    private function inject_youtube( string $content, string $video_id ): string {
        // Sanitize video ID (only alphanumeric and dashes/underscores)
        $video_id = preg_replace( '/[^a-zA-Z0-9_-]/', '', $video_id );

        if ( empty( $video_id ) ) {
            return $content;
        }

        // Check if content already contains this YouTube video (avoid duplication)
        if ( strpos( $content, 'youtube.com/embed/' . $video_id ) !== false ) {
            return $content;
        }

        // Check if content already contains any YouTube embed
        if ( strpos( $content, 'youtube.com/embed/' ) !== false ) {
            return $content;
        }

        // Create responsive YouTube embed
        $youtube_embed = sprintf(
            '<div class="seoauto-youtube-embed">
                <div class="seoauto-youtube-wrapper">
                    <iframe
                        src="https://www.youtube.com/embed/%s"
                        title="YouTube video player"
                        frameborder="0"
                        allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share"
                        allowfullscreen
                        loading="lazy"
                    ></iframe>
                </div>
            </div>',
            esc_attr( $video_id )
        );

        // Find second heading (h2, h3) and inject before it
        $pattern = '/<h[23][^>]*>/i';

        if ( preg_match( $pattern, $content, $matches, PREG_OFFSET_CAPTURE, 100 ) ) {
            // Found a heading after first 100 chars, inject before it
            $pos     = $matches[0][1];
            $content = substr( $content, 0, $pos ) . $youtube_embed . substr( $content, $pos );
        } else {
            // No suitable heading found, inject after first paragraph
            $pos = strpos( $content, '</p>' );

            if ( false !== $pos ) {
                $content = substr_replace( $content, '</p>' . $youtube_embed, $pos, 4 );
            } else {
                // Prepend if no paragraph
                $content = $youtube_embed . $content;
            }
        }

        return $content;
    }

    /**
     * Count words in content.
     *
     * @param string $content The content.
     * @return int Word count.
     */
    public function count_words( string $content ): int {
        // Strip HTML tags
        $text = wp_strip_all_tags( $content );

        // Count words
        return str_word_count( $text );
    }

    /**
     * Generate excerpt from content.
     *
     * @param string $content The content.
     * @param int    $length  Maximum excerpt length.
     * @return string Generated excerpt.
     */
    public function generate_excerpt( string $content, int $length = 160 ): string {
        // Strip HTML
        $text = wp_strip_all_tags( $content );

        // Trim to length
        if ( strlen( $text ) > $length ) {
            $text = substr( $text, 0, $length );
            $text = substr( $text, 0, strrpos( $text, ' ' ) );
            $text = rtrim( $text, '.,;:!?' ) . '...';
        }

        return $text;
    }

    /**
     * Clean up HTML content.
     *
     * @param string $content The content.
     * @return string Cleaned content.
     */
    public function clean_html( string $content ): string {
        // Remove empty paragraphs
        $content = preg_replace( '/<p>\s*<\/p>/i', '', $content );

        // Remove multiple consecutive line breaks
        $content = preg_replace( '/(\r\n|\n|\r){3,}/', "\n\n", $content );

        // Remove excessive whitespace
        $content = preg_replace( '/\s+/', ' ', $content );

        // Fix broken tags
        $content = balanceTags( $content, true );

        return trim( $content );
    }
}
