<?php
/**
 * Article Publisher Class
 *
 * @package SEOAuto\Plugin\Publisher
 */

namespace SEOAuto\Plugin\Publisher;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SEOAuto\Plugin\Api\Client;
use SEOAuto\Plugin\Support\Logger;

/**
 * Handles publishing articles to WordPress.
 *
 * @since 1.0.0
 */
class ArticlePublisher {

    /**
     * Image handler.
     *
     * @var ImageHandler
     */
    private ImageHandler $image_handler;

    /**
     * Content processor.
     *
     * @var ContentProcessor
     */
    private ContentProcessor $content_processor;

    /**
     * SEO integration.
     *
     * @var SeoIntegration
     */
    private SeoIntegration $seo_integration;

    /**
     * Constructor.
     */
    public function __construct() {
        $this->image_handler     = new ImageHandler();
        $this->content_processor = new ContentProcessor();
        $this->seo_integration   = new SeoIntegration();
    }

    /**
     * Publish article to WordPress.
     *
     * @param array $article_data Data from SEOAuto API.
     * @return array{post_id: int, permalink: string} Published post info.
     * @throws \Exception When publishing fails.
     */
    public function publish( array $article_data ): array {
        Logger::debug( 'Publishing article', array( 'seoauto_id' => $article_data['seoauto_id'] ?? '' ) );

        // Check for existing article (prevent duplicates)
        $existing_post_id = $this->find_existing_post( $article_data['seoauto_id'] );

        if ( $existing_post_id ) {
            return $this->update_post( $existing_post_id, $article_data );
        }

        return $this->create_post( $article_data );
    }

    /**
     * Create new WordPress post.
     *
     * @param array $data Article data.
     * @return array{post_id: int, permalink: string} Created post info.
     * @throws \Exception When creation fails.
     */
    private function create_post( array $data ): array {
        // Process content (inject infographic, YouTube, etc.)
        $processed_content = $this->content_processor->process(
            $data['content'],
            array(
                'infographic_html'  => $data['infographic_html'] ?? null,
                'youtube_video_id'  => $data['youtube_video_id'] ?? null,
            )
        );

        // Get default settings
        $default_category = (int) get_option( 'seoauto_default_category', 1 );
        $default_author   = (int) get_option( 'seoauto_default_author', 1 );
        $post_status      = get_option( 'seoauto_post_status', 'publish' );

        // Verify author exists
        $author = get_user_by( 'id', $default_author );
        if ( ! $author ) {
            $default_author = 1; // Fallback to admin
        }

        // Verify category exists
        if ( ! term_exists( $default_category, 'category' ) ) {
            $default_category = 1; // Fallback to uncategorized
        }

        /**
         * Filter the post status before publishing.
         *
         * @param string $post_status The post status.
         * @param array  $data        The article data.
         */
        $post_status = apply_filters( 'seoauto_post_status', $post_status, $data );

        /**
         * Filter the post author before publishing.
         *
         * @param int   $default_author The author ID.
         * @param array $data           The article data.
         */
        $default_author = apply_filters( 'seoauto_post_author', $default_author, $data );

        /**
         * Filter the post category before publishing.
         *
         * @param int   $default_category The category ID.
         * @param array $data             The article data.
         */
        $default_category = apply_filters( 'seoauto_post_category', $default_category, $data );

        // Prepare post data
        $post_data = array(
            'post_title'    => $data['title'],
            'post_content'  => $processed_content,
            'post_excerpt'  => $data['excerpt'] ?? '',
            'post_status'   => $post_status,
            'post_author'   => $default_author,
            'post_category' => array( $default_category ),
            'meta_input'    => array(
                '_seoauto_article_id' => $data['seoauto_id'],
                '_seoauto_managed'    => '1',
                '_seoauto_synced_at'  => current_time( 'mysql' ),
            ),
        );

        /**
         * Filter the post data before inserting.
         *
         * @param array $post_data The post data.
         * @param array $data      The article data from API.
         */
        $post_data = apply_filters( 'seoauto_post_data', $post_data, $data );

        // Temporarily allow iframes (YouTube embeds) in post content
        // WordPress kses normally strips iframe tags for security
        kses_remove_filters();

        // Insert post
        $post_id = wp_insert_post( $post_data, true );

        // Restore kses filters
        kses_init_filters();

        if ( is_wp_error( $post_id ) ) {
            throw new \Exception( esc_html( $post_id->get_error_message() ) );
        }

        // Handle featured image
        if ( ! empty( $data['featured_image_url'] ) ) {
            $image_id = $this->image_handler->download_and_attach(
                $data['featured_image_url'],
                $post_id,
                $data['title'] . ' - Featured Image'
            );

            if ( $image_id && ! is_wp_error( $image_id ) ) {
                set_post_thumbnail( $post_id, $image_id );

                do_action( 'seoauto_image_uploaded', $image_id, $post_id );
            } else {
                $error_msg = is_wp_error( $image_id ) ? $image_id->get_error_message() : 'Unknown error';
                Logger::warning( 'Failed to download featured image', array( 'error' => $error_msg ) );

                do_action( 'seoauto_image_failed', $data['featured_image_url'], $error_msg );
            }
        }

        // Handle tags
        if ( ! empty( $data['tags'] ) ) {
            wp_set_post_tags( $post_id, $data['tags'] );
        }

        // SEO Integration (Yoast, Rank Math, etc.)
        $this->seo_integration->set_metadata(
            $post_id,
            array(
                'meta_description' => $data['meta_description'] ?? '',
                'focus_keyword'    => $data['focus_keyword'] ?? '',
                'meta_keywords'    => $data['meta_keywords'] ?? '',
            )
        );

        // Store in local tracking table
        $this->save_article_record( $data['seoauto_id'], $post_id, 'published' );

        // Notify SEOAuto backend
        $this->notify_published( $data['seoauto_id'], $post_id );

        Logger::info(
            'Article published',
            array(
                'seoauto_id' => $data['seoauto_id'],
                'post_id'     => $post_id,
            )
        );

        return array(
            'post_id'   => $post_id,
            'permalink' => get_permalink( $post_id ),
        );
    }

    /**
     * Update existing WordPress post.
     *
     * @param int   $post_id The WordPress post ID.
     * @param array $data    Article data.
     * @return array{post_id: int, permalink: string} Updated post info.
     * @throws \Exception When update fails.
     */
    private function update_post( int $post_id, array $data ): array {
        $processed_content = $this->content_processor->process(
            $data['content'],
            array(
                'infographic_html' => $data['infographic_html'] ?? null,
                'youtube_video_id' => $data['youtube_video_id'] ?? null,
            )
        );

        $post_data = array(
            'ID'           => $post_id,
            'post_title'   => $data['title'],
            'post_content' => $processed_content,
            'post_excerpt' => $data['excerpt'] ?? '',
        );

        // Temporarily allow iframes (YouTube embeds) in post content
        kses_remove_filters();

        $result = wp_update_post( $post_data, true );

        // Restore kses filters
        kses_init_filters();

        if ( is_wp_error( $result ) ) {
            throw new \Exception( esc_html( $result->get_error_message() ) );
        }

        // Update featured image if provided and different
        if ( ! empty( $data['featured_image_url'] ) ) {
            $current_url = get_post_meta( $post_id, '_seoauto_featured_image_url', true );

            if ( $current_url !== $data['featured_image_url'] ) {
                $image_id = $this->image_handler->download_and_attach(
                    $data['featured_image_url'],
                    $post_id,
                    $data['title'] . ' - Featured Image'
                );

                if ( $image_id && ! is_wp_error( $image_id ) ) {
                    set_post_thumbnail( $post_id, $image_id );
                    update_post_meta( $post_id, '_seoauto_featured_image_url', $data['featured_image_url'] );
                }
            }
        }

        // Update tags
        if ( ! empty( $data['tags'] ) ) {
            wp_set_post_tags( $post_id, $data['tags'] );
        }

        // Update SEO metadata
        $this->seo_integration->set_metadata(
            $post_id,
            array(
                'meta_description' => $data['meta_description'] ?? '',
                'focus_keyword'    => $data['focus_keyword'] ?? '',
            )
        );

        // Update tracking record
        $this->update_article_record( $data['seoauto_id'], 'published' );

        // Update synced timestamp
        update_post_meta( $post_id, '_seoauto_synced_at', current_time( 'mysql' ) );

        Logger::info(
            'Article updated',
            array(
                'seoauto_id' => $data['seoauto_id'],
                'post_id'     => $post_id,
            )
        );

        return array(
            'post_id'   => $post_id,
            'permalink' => get_permalink( $post_id ),
        );
    }

    /**
     * Find existing post by SEOAuto ID.
     *
     * @param string $seoauto_id The SEOAuto article ID.
     * @return int|null The post ID or null if not found.
     */
    private function find_existing_post( string $seoauto_id ): ?int {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time lookup, caching not needed.
        $post_id = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT post_id FROM {$wpdb->postmeta}
                WHERE meta_key = '_seoauto_article_id' AND meta_value = %s
                LIMIT 1",
                $seoauto_id
            )
        );

        return $post_id ? (int) $post_id : null;
    }

    /**
     * Save article record to tracking table.
     *
     * @param string $seoauto_id The SEOAuto article ID.
     * @param int    $post_id     The WordPress post ID.
     * @param string $status      The article status.
     * @return void
     */
    private function save_article_record( string $seoauto_id, int $post_id, string $status ): void {
        global $wpdb;

        $table = $wpdb->prefix . 'seoauto_articles';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table insert/update.
        $wpdb->replace(
            $table,
            array(
                'seoauto_id'  => $seoauto_id,
                'post_id'      => $post_id,
                'title'        => get_the_title( $post_id ),
                'status'       => $status,
                'synced_at'    => current_time( 'mysql' ),
                'published_at' => 'published' === $status ? current_time( 'mysql' ) : null,
            ),
            array( '%s', '%d', '%s', '%s', '%s', '%s' )
        );
    }

    /**
     * Update article record status.
     *
     * @param string $seoauto_id The SEOAuto article ID.
     * @param string $status      The new status.
     * @return void
     */
    private function update_article_record( string $seoauto_id, string $status ): void {
        global $wpdb;

        $table = $wpdb->prefix . 'seoauto_articles';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update.
        $wpdb->update(
            $table,
            array(
                'status'    => $status,
                'synced_at' => current_time( 'mysql' ),
            ),
            array( 'seoauto_id' => $seoauto_id ),
            array( '%s', '%s' ),
            array( '%s' )
        );
    }

    /**
     * Notify SEOAuto backend that article was published.
     *
     * @param string $seoauto_id The SEOAuto article ID.
     * @param int    $post_id     The WordPress post ID.
     * @return void
     */
    private function notify_published( string $seoauto_id, int $post_id ): void {
        // Skip in stub mode
        if ( defined( 'SEOAUTO_STUB_MODE' ) && SEOAUTO_STUB_MODE ) {
            Logger::debug( 'Skipping publish notification in stub mode' );
            return;
        }

        try {
            $client = new Client();

            $client->post(
                '/wp/articles/' . $seoauto_id . '/published',
                array(
                    'wordpress_post_id' => $post_id,
                    'permalink'         => get_permalink( $post_id ),
                    'published_at'      => current_time( 'c' ),
                )
            );

            Logger::debug( 'Publish notification sent', array( 'seoauto_id' => $seoauto_id ) );

        } catch ( \Exception $e ) {
            // Log but don't fail - webhook can be retried
            Logger::warning(
                'Failed to notify published',
                array(
                    'seoauto_id' => $seoauto_id,
                    'error'       => $e->getMessage(),
                )
            );
        }
    }
}
