<?php
/**
 * Main Plugin Class
 *
 * @package SEOAuto\Plugin
 */

namespace SEOAuto\Plugin;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SEOAuto\Plugin\Admin\AdminMenu;
use SEOAuto\Plugin\Admin\SettingsPage;
use SEOAuto\Plugin\Admin\Notices;
use SEOAuto\Plugin\Publisher\PostEventHandler;
use SEOAuto\Plugin\Rest\RestController;
use SEOAuto\Plugin\Scheduler\CronManager;
use SEOAuto\Plugin\Support\Logger;

/**
 * Main plugin class using Singleton pattern.
 *
 * @since 1.0.0
 */
final class Plugin {

    /**
     * Plugin instance.
     *
     * @var Plugin|null
     */
    private static ?Plugin $instance = null;

    /**
     * Admin menu instance.
     *
     * @var AdminMenu|null
     */
    private ?AdminMenu $admin_menu = null;

    /**
     * Settings page instance.
     *
     * @var SettingsPage|null
     */
    private ?SettingsPage $settings_page = null;

    /**
     * REST controller instance.
     *
     * @var RestController|null
     */
    private ?RestController $rest_controller = null;

    /**
     * Cron manager instance.
     *
     * @var CronManager|null
     */
    private ?CronManager $cron_manager = null;

    /**
     * Admin notices instance.
     *
     * @var Notices|null
     */
    private ?Notices $notices = null;

    /**
     * Post event handler instance.
     *
     * @var PostEventHandler|null
     */
    private ?PostEventHandler $post_event_handler = null;

    /**
     * Get plugin instance.
     *
     * @return Plugin
     */
    public static function get_instance(): Plugin {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private constructor to prevent direct instantiation.
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Prevent cloning.
     *
     * @return void
     */
    private function __clone() {}

    /**
     * Prevent unserialization.
     *
     * @return void
     * @throws \Exception When trying to unserialize.
     */
    public function __wakeup() {
        throw new \Exception( 'Cannot unserialize singleton' );
    }

    /**
     * Initialize plugin hooks.
     *
     * @return void
     */
    private function init_hooks(): void {
        // Initialize components after WordPress is ready
        add_action( 'init', array( $this, 'init' ), 0 );

        // Admin hooks
        if ( is_admin() ) {
            add_action( 'admin_init', array( $this, 'admin_init' ) );
            add_action( 'admin_menu', array( $this, 'admin_menu' ) );
            add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
        }

        // REST API
        add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );

        // Plugin action links
        add_filter( 'plugin_action_links_' . SEOAUTO_PLUGIN_BASENAME, array( $this, 'plugin_action_links' ) );

        // Cron hooks
        add_action( 'seoauto_auto_sync', array( $this, 'run_auto_sync' ) );
        add_action( 'seoauto_cleanup', array( $this, 'run_cleanup' ) );
        add_action( 'seoauto_health_check', array( $this, 'run_health_check' ) );
    }

    /**
     * Initialize plugin.
     *
     * @return void
     */
    public function init(): void {
        // Fire action for other plugins/themes to hook into
        do_action( 'seoauto_init' );

        // Initialize cron manager
        $this->cron_manager = new CronManager();
        $this->cron_manager->register_callbacks();

        // Initialize post event handler for delete/trash/restore notifications
        $this->post_event_handler = new PostEventHandler();
        $this->post_event_handler->register_hooks();

        Logger::debug( 'Plugin initialized' );
    }

    /**
     * Admin initialization.
     *
     * @return void
     */
    public function admin_init(): void {
        // Initialize settings
        $this->settings_page = new SettingsPage();
        $this->settings_page->register();

        // Initialize notices
        $this->notices = new Notices();

        // Check for database updates
        $this->maybe_update_database();

        do_action( 'seoauto_admin_init' );
    }

    /**
     * Register admin menu.
     *
     * @return void
     */
    public function admin_menu(): void {
        $this->admin_menu = new AdminMenu();
        $this->admin_menu->register();
    }

    /**
     * Enqueue admin scripts and styles.
     *
     * @param string $hook_suffix The current admin page hook suffix.
     * @return void
     */
    public function admin_enqueue_scripts( string $hook_suffix ): void {
        // Only load on SEOAuto pages (check for 'seoauto' in hook suffix)
        if ( strpos( $hook_suffix, 'seoauto' ) === false ) {
            return;
        }

        // Enqueue styles
        wp_enqueue_style(
            'seoauto-admin',
            SEOAUTO_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            SEOAUTO_VERSION
        );

        // Enqueue scripts
        wp_enqueue_script(
            'seoauto-admin',
            SEOAUTO_PLUGIN_URL . 'assets/js/admin.js',
            array( 'jquery' ),
            SEOAUTO_VERSION,
            true
        );

        // Localize script
        wp_localize_script(
            'seoauto-admin',
            'seoautoAdmin',
            array(
                'ajaxUrl'    => admin_url( 'admin-ajax.php' ),
                'restUrl'    => rest_url( 'seoauto/v1/' ),
                'nonce'      => wp_create_nonce( 'seoauto_admin_nonce' ),
                'restNonce'  => wp_create_nonce( 'wp_rest' ),
                'pluginUrl'  => SEOAUTO_PLUGIN_URL,
                'version'    => SEOAUTO_VERSION,
                'debug'      => $this->is_debug_mode(),
                'strings'    => array(
                    'confirmSync'    => __( 'Are you sure you want to sync articles now?', 'seoauto-ai-content-publisher' ),
                    'syncSuccess'    => __( 'Articles synced successfully!', 'seoauto-ai-content-publisher' ),
                    'syncError'      => __( 'Failed to sync articles. Please try again.', 'seoauto-ai-content-publisher' ),
                    'saving'         => __( 'Saving...', 'seoauto-ai-content-publisher' ),
                    'saved'          => __( 'Settings saved!', 'seoauto-ai-content-publisher' ),
                    'error'          => __( 'An error occurred. Please try again.', 'seoauto-ai-content-publisher' ),
                    'confirmDelete'  => __( 'Are you sure you want to delete this item?', 'seoauto-ai-content-publisher' ),
                    'validating'     => __( 'Validating API key...', 'seoauto-ai-content-publisher' ),
                    'validKey'       => __( 'API key is valid!', 'seoauto-ai-content-publisher' ),
                    'invalidKey'     => __( 'Invalid API key. Please check and try again.', 'seoauto-ai-content-publisher' ),
                ),
            )
        );
    }

    /**
     * Register REST API routes.
     *
     * @return void
     */
    public function register_rest_routes(): void {
        $this->rest_controller = new RestController();
        $this->rest_controller->register_routes();
    }

    /**
     * Add plugin action links.
     *
     * @param array $links Existing plugin action links.
     * @return array Modified plugin action links.
     */
    public function plugin_action_links( array $links ): array {
        $plugin_links = array(
            '<a href="' . admin_url( 'admin.php?page=seoauto-settings' ) . '">' . __( 'Settings', 'seoauto-ai-content-publisher' ) . '</a>',
            '<a href="' . admin_url( 'admin.php?page=seoauto' ) . '">' . __( 'Dashboard', 'seoauto-ai-content-publisher' ) . '</a>',
        );

        return array_merge( $plugin_links, $links );
    }

    /**
     * Run auto sync cron job.
     *
     * @return void
     */
    public function run_auto_sync(): void {
        if ( $this->cron_manager ) {
            $this->cron_manager->run_auto_sync();
        }
    }

    /**
     * Run cleanup cron job.
     *
     * @return void
     */
    public function run_cleanup(): void {
        if ( $this->cron_manager ) {
            $this->cron_manager->run_cleanup();
        }
    }

    /**
     * Run health check cron job.
     *
     * @return void
     */
    public function run_health_check(): void {
        if ( $this->cron_manager ) {
            $this->cron_manager->run_health_check();
        }
    }

    /**
     * Check if debug mode is enabled.
     *
     * @return bool
     */
    public function is_debug_mode(): bool {
        return (bool) get_option( 'seoauto_debug_mode', false );
    }

    /**
     * Get API key.
     *
     * @return string|null
     */
    public function get_api_key(): ?string {
        $api_key = get_option( 'seoauto_api_key', '' );
        return ! empty( $api_key ) ? $api_key : null;
    }

    /**
     * Check if plugin is connected to SEOAuto API.
     * This checks both credentials exist AND last validation was successful.
     *
     * @return bool
     */
    public function is_connected(): bool {
        $api_key = $this->get_api_key();
        $site_id = get_option( 'seoauto_site_id', '' );

        // Must have both API key and site ID
        if ( empty( $api_key ) || empty( $site_id ) ) {
            return false;
        }

        // Check if connection was validated (default to true for backward compatibility)
        $connection_valid = get_option( 'seoauto_connection_valid', true );

        return (bool) $connection_valid;
    }

    /**
     * Set connection validation status.
     *
     * @param bool $valid Whether connection is valid.
     * @return void
     */
    public function set_connection_status( bool $valid ): void {
        update_option( 'seoauto_connection_valid', $valid, false );

        if ( $valid ) {
            update_option( 'seoauto_last_successful_validation', current_time( 'mysql' ), false );
        } else {
            // Clear site ID when connection fails
            delete_option( 'seoauto_site_id' );
        }
    }

    /**
     * Get connection error message if any.
     *
     * @return string|null
     */
    public function get_connection_error(): ?string {
        return get_option( 'seoauto_connection_error', null );
    }

    /**
     * Set connection error message.
     *
     * @param string|null $error Error message or null to clear.
     * @return void
     */
    public function set_connection_error( ?string $error ): void {
        if ( $error ) {
            update_option( 'seoauto_connection_error', $error, false );
        } else {
            delete_option( 'seoauto_connection_error' );
        }
    }

    /**
     * Maybe update database schema.
     *
     * @return void
     */
    private function maybe_update_database(): void {
        $current_version = get_option( 'seoauto_db_version', '0' );

        if ( version_compare( $current_version, SEOAUTO_VERSION, '<' ) ) {
            Activator::update_database();
            update_option( 'seoauto_db_version', SEOAUTO_VERSION );

            Logger::info( 'Database updated to version ' . SEOAUTO_VERSION );
        }
    }

    /**
     * Check if a database table exists.
     *
     * @param string $table_name Table name without prefix.
     * @return bool True if table exists.
     */
    public static function table_exists( string $table_name ): bool {
        global $wpdb;

        $full_table_name = $wpdb->prefix . $table_name;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table existence check.
        return $wpdb->get_var(
            $wpdb->prepare( 'SHOW TABLES LIKE %s', $full_table_name )
        ) === $full_table_name;
    }

    /**
     * Get plugin version.
     *
     * @return string
     */
    public function get_version(): string {
        return SEOAUTO_VERSION;
    }

    /**
     * Get plugin directory path.
     *
     * @return string
     */
    public function get_plugin_dir(): string {
        return SEOAUTO_PLUGIN_DIR;
    }

    /**
     * Get plugin URL.
     *
     * @return string
     */
    public function get_plugin_url(): string {
        return SEOAUTO_PLUGIN_URL;
    }
}
