<?php
/**
 * Plugin Deactivator
 *
 * @package SEOAuto\Plugin
 */

namespace SEOAuto\Plugin;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles plugin deactivation.
 *
 * @since 1.0.0
 */
class Deactivator {

    /**
     * Run deactivation routines.
     *
     * @return void
     */
    public static function deactivate(): void {
        // Clear all scheduled cron events
        self::clear_cron_events();

        // Clear transients
        self::clear_transients();

        // Flush rewrite rules
        flush_rewrite_rules();

        // Fire deactivation action
        do_action( 'seoauto_deactivated' );
    }

    /**
     * Clear all scheduled cron events.
     *
     * @return void
     */
    private static function clear_cron_events(): void {
        $cron_hooks = array(
            'seoauto_auto_sync',
            'seoauto_cleanup',
            'seoauto_health_check',
        );

        foreach ( $cron_hooks as $hook ) {
            $timestamp = wp_next_scheduled( $hook );
            if ( $timestamp ) {
                wp_unschedule_event( $timestamp, $hook );
            }
            wp_clear_scheduled_hook( $hook );
        }
    }

    /**
     * Clear plugin transients.
     *
     * @return void
     */
    private static function clear_transients(): void {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup on deactivation.
        $wpdb->query(
            "DELETE FROM {$wpdb->options}
            WHERE option_name LIKE '_transient_seoauto_%'
            OR option_name LIKE '_transient_timeout_seoauto_%'"
        );

        // Delete site transients if multisite
        if ( is_multisite() ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup on deactivation.
            $wpdb->query(
                "DELETE FROM {$wpdb->sitemeta}
                WHERE meta_key LIKE '_site_transient_seoauto_%'
                OR meta_key LIKE '_site_transient_timeout_seoauto_%'"
            );
        }
    }
}
