<?php
/**
 * API Client Class
 *
 * @package SEOAuto\Plugin\Api
 */

namespace SEOAuto\Plugin\Api;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SEOAuto\Plugin\Support\Logger;

/**
 * HTTP client for communicating with SEOAuto API.
 *
 * @since 1.0.0
 */
class Client {

    /**
     * Default request timeout in seconds.
     */
    private const DEFAULT_TIMEOUT = 30;

    /**
     * API base URL.
     *
     * @var string
     */
    private string $base_url;

    /**
     * API key for authentication.
     *
     * @var string|null
     */
    private ?string $api_key;

    /**
     * Constructor.
     */
    public function __construct() {
        $this->base_url = SEOAUTO_API_BASE_URL;
        $this->api_key  = get_option( 'seoauto_api_key', '' ) ?: null;
    }

    /**
     * Make a GET request.
     *
     * @param string $endpoint The API endpoint.
     * @param array  $params   Query parameters.
     * @return array Response data.
     * @throws \Exception When request fails.
     */
    public function get( string $endpoint, array $params = array() ): array {
        return $this->request( 'GET', $endpoint, array( 'query' => $params ) );
    }

    /**
     * Make a POST request.
     *
     * @param string $endpoint The API endpoint.
     * @param array  $data     Request body data.
     * @return array Response data.
     * @throws \Exception When request fails.
     */
    public function post( string $endpoint, array $data = array() ): array {
        return $this->request( 'POST', $endpoint, array( 'body' => $data ) );
    }

    /**
     * Make a PATCH request.
     *
     * @param string $endpoint The API endpoint.
     * @param array  $data     Request body data.
     * @return array Response data.
     * @throws \Exception When request fails.
     */
    public function patch( string $endpoint, array $data = array() ): array {
        return $this->request( 'PATCH', $endpoint, array( 'body' => $data ) );
    }

    /**
     * Make a DELETE request.
     *
     * @param string $endpoint The API endpoint.
     * @return array Response data.
     * @throws \Exception When request fails.
     */
    public function delete( string $endpoint ): array {
        return $this->request( 'DELETE', $endpoint );
    }

    /**
     * Make an HTTP request.
     *
     * @param string $method   HTTP method.
     * @param string $endpoint API endpoint.
     * @param array  $options  Request options.
     * @return array Response data.
     * @throws \Exception When request fails.
     */
    private function request( string $method, string $endpoint, array $options = array() ): array {
        $url = $this->base_url . $endpoint;

        // Add query parameters for GET requests
        if ( 'GET' === $method && ! empty( $options['query'] ) ) {
            $url = add_query_arg( $options['query'], $url );
        }

        /**
         * Filter the API request timeout.
         *
         * @param int $timeout Default timeout in seconds.
         */
        $timeout = apply_filters( 'seoauto_api_timeout', self::DEFAULT_TIMEOUT );

        $headers = array(
            'Content-Type'               => 'application/json',
            'Accept'                     => 'application/json',
            'X-SEOAuto-Plugin-Version'  => SEOAUTO_VERSION,
            'X-WordPress-Site-URL'       => site_url(),
            'X-WordPress-Version'        => get_bloginfo( 'version' ),
        );

        // Add authorization header if API key is available
        if ( $this->api_key ) {
            $headers['Authorization'] = 'Bearer ' . $this->api_key;
        }

        /**
         * Filter the API request headers.
         *
         * @param array $headers Request headers.
         */
        $headers = apply_filters( 'seoauto_api_headers', $headers );

        // Disable SSL verification for localhost development
        $is_localhost = strpos( $this->base_url, 'localhost' ) !== false
                     || strpos( $this->base_url, '127.0.0.1' ) !== false;

        $args = array(
            'method'    => $method,
            'headers'   => $headers,
            'timeout'   => $timeout,
            'sslverify' => ! $is_localhost,
        );

        // Add body for non-GET requests
        if ( 'GET' !== $method && ! empty( $options['body'] ) ) {
            $args['body'] = wp_json_encode( $options['body'] );
        }

        Logger::debug(
            'API Request',
            array(
                'method'   => $method,
                'endpoint' => $endpoint,
                'url'      => $url,
            )
        );

        $response = wp_remote_request( $url, $args );

        if ( is_wp_error( $response ) ) {
            Logger::error(
                'API Request Failed',
                array(
                    'endpoint' => $endpoint,
                    'error'    => $response->get_error_message(),
                )
            );

            throw new \Exception( esc_html( $response->get_error_message() ) );
        }

        $status_code = wp_remote_retrieve_response_code( $response );
        $body        = wp_remote_retrieve_body( $response );
        $data        = json_decode( $body, true );

        // Handle JSON decode errors
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            Logger::error(
                'Invalid JSON response from API',
                array(
                    'endpoint'   => $endpoint,
                    'json_error' => json_last_error_msg(),
                    'body'       => substr( $body, 0, 500 ), // Log first 500 chars for debugging
                )
            );

            throw new \Exception( esc_html__( 'Invalid JSON response from API', 'seoauto-ai-content-publisher' ) );
        }

        Logger::debug(
            'API Response',
            array(
                'endpoint'    => $endpoint,
                'status_code' => $status_code,
            )
        );

        // Handle error responses
        if ( $status_code >= 400 ) {
            $error_message = $data['message'] ?? __( 'API request failed', 'seoauto-ai-content-publisher' );

            Logger::error(
                'API Error Response',
                array(
                    'endpoint'    => $endpoint,
                    'status_code' => $status_code,
                    'message'     => $error_message,
                )
            );

            throw new \Exception( esc_html( $error_message ) );
        }

        return $data ?: array();
    }

    /**
     * Test API connection.
     *
     * @return bool True if connection is successful.
     */
    public function test_connection(): bool {
        try {
            $auth = new Authentication();
            $auth->validate_api_key( $this->api_key );
            return true;
        } catch ( \Exception $e ) {
            return false;
        }
    }

    /**
     * Set API key.
     *
     * @param string $api_key The API key.
     * @return void
     */
    public function set_api_key( string $api_key ): void {
        $this->api_key = $api_key;
    }
}
