<?php
/**
 * Authentication Class
 *
 * @package SEOAuto\Plugin\Api
 */

namespace SEOAuto\Plugin\Api;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles API key authentication with SEOAuto backend.
 *
 * @since 1.0.0
 */
class Authentication {

    /**
     * Option name for API key.
     */
    private const OPTION_API_KEY = 'seoauto_api_key';

    /**
     * Option name for site ID.
     */
    private const OPTION_SITE_ID = 'seoauto_site_id';

    /**
     * Validate API key with SEOAuto backend.
     *
     * @param string $api_key The API key to validate.
     * @return array Validation response data.
     * @throws \Exception When validation fails.
     */
    public function validate_api_key( string $api_key ): array {
        // STUB MODE: For development without backend
        if ( defined( 'SEOAUTO_STUB_MODE' ) && SEOAUTO_STUB_MODE ) {
            return $this->get_stub_validation_response();
        }

        // Disable SSL verification for localhost development
        $is_localhost = strpos( SEOAUTO_API_BASE_URL, 'localhost' ) !== false
                     || strpos( SEOAUTO_API_BASE_URL, '127.0.0.1' ) !== false;

        $response = wp_remote_post(
            SEOAUTO_API_BASE_URL . '/wp/validate',
            array(
                'headers'   => array(
                    'Authorization'              => 'Bearer ' . $api_key,
                    'Content-Type'               => 'application/json',
                    'X-SEOAuto-Plugin-Version'  => SEOAUTO_VERSION,
                    'X-WordPress-Site-URL'       => site_url(),
                    'X-WordPress-Version'        => get_bloginfo( 'version' ),
                ),
                'timeout'   => 30,
                'sslverify' => ! $is_localhost,
            )
        );

        if ( is_wp_error( $response ) ) {
            throw new \Exception( esc_html( $response->get_error_message() ) );
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );
        $code = wp_remote_retrieve_response_code( $response );

        if ( 200 !== $code ) {
            throw new \Exception( esc_html( $body['message'] ?? __( 'Invalid API key', 'seoauto-ai-content-publisher' ) ) );
        }

        return $body;
    }

    /**
     * Store encrypted API key.
     *
     * @param string $api_key The API key to store.
     * @param string $site_id The site ID from validation response.
     * @return void
     */
    public function save_credentials( string $api_key, string $site_id ): void {
        // For now, store the API key as-is (encryption can be added later)
        // The key is already protected by WordPress database security
        update_option( self::OPTION_API_KEY, $api_key, false );
        update_option( self::OPTION_SITE_ID, $site_id, false );
    }

    /**
     * Get stored API key.
     *
     * @return string|null The API key or null if not set.
     */
    public function get_api_key(): ?string {
        $api_key = get_option( self::OPTION_API_KEY, '' );
        return ! empty( $api_key ) ? $api_key : null;
    }

    /**
     * Get stored site ID.
     *
     * @return string|null The site ID or null if not set.
     */
    public function get_site_id(): ?string {
        $site_id = get_option( self::OPTION_SITE_ID, '' );
        return ! empty( $site_id ) ? $site_id : null;
    }

    /**
     * Clear stored credentials.
     *
     * @return void
     */
    public function clear_credentials(): void {
        delete_option( self::OPTION_API_KEY );
        delete_option( self::OPTION_SITE_ID );
        delete_option( 'seoauto_connected_at' );
    }

    /**
     * Check if plugin is authenticated.
     *
     * @return bool
     */
    public function is_authenticated(): bool {
        return ! empty( $this->get_api_key() ) && ! empty( $this->get_site_id() );
    }

    /**
     * Get stub validation response for development.
     *
     * @return array
     */
    private function get_stub_validation_response(): array {
        return array(
            'success'         => true,
            'site_id'         => 'site_stub_' . substr( md5( site_url() ), 0, 8 ),
            'website_name'    => get_bloginfo( 'name' ),
            'website_url'     => site_url(),
            'plan'            => 'professional',
            'articles_quota'  => 30,
            'articles_used'   => 0,
            'backlinks_quota' => 10,
            'backlinks_used'  => 0,
        );
    }
}
