<?php
/**
 * Settings Page Class
 *
 * @package SEOAuto\Plugin\Admin
 */

namespace SEOAuto\Plugin\Admin;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SEOAuto\Plugin\Api\Authentication;

/**
 * Handles plugin settings registration and validation.
 *
 * @since 1.0.0
 */
class SettingsPage {

    /**
     * Option group name.
     *
     * @var string
     */
    private const OPTION_GROUP = 'seoauto_settings';

    /**
     * Register settings.
     *
     * @return void
     */
    public function register(): void {
        // Connection settings
        register_setting(
            self::OPTION_GROUP,
            'seoauto_api_key',
            array(
                'type'              => 'string',
                'sanitize_callback' => array( $this, 'sanitize_api_key' ),
                'default'           => '',
            )
        );

        // Publishing settings
        register_setting(
            self::OPTION_GROUP,
            'seoauto_default_category',
            array(
                'type'              => 'integer',
                'sanitize_callback' => 'absint',
                'default'           => 1,
            )
        );

        register_setting(
            self::OPTION_GROUP,
            'seoauto_default_author',
            array(
                'type'              => 'integer',
                'sanitize_callback' => 'absint',
                'default'           => 1,
            )
        );

        // Debug mode
        register_setting(
            self::OPTION_GROUP,
            'seoauto_debug_mode',
            array(
                'type'              => 'boolean',
                'sanitize_callback' => 'rest_sanitize_boolean',
                'default'           => false,
            )
        );

        // Register AJAX handlers
        add_action( 'wp_ajax_seoauto_validate_api_key', array( $this, 'ajax_validate_api_key' ) );
        add_action( 'wp_ajax_seoauto_test_connection', array( $this, 'ajax_test_connection' ) );
        add_action( 'wp_ajax_seoauto_reset_plugin', array( $this, 'ajax_reset_plugin' ) );
    }

    /**
     * Sanitize and validate API key.
     *
     * @param string $value The API key value.
     * @return string Sanitized value.
     */
    public function sanitize_api_key( string $value ): string {
        $value  = sanitize_text_field( $value );
        $plugin = \SEOAuto\Plugin\Plugin::get_instance();

        if ( empty( $value ) ) {
            // Clear all connection data when API key is removed
            delete_option( 'seoauto_site_id' );
            delete_option( 'seoauto_connected_at' );
            delete_option( 'seoauto_website_name' );
            delete_option( 'seoauto_website_url' );
            $plugin->set_connection_status( false );
            $plugin->set_connection_error( null );
            return '';
        }

        // Validate with SEOAuto API
        try {
            $auth   = new Authentication();
            $result = $auth->validate_api_key( $value );

            // Store site info from response
            if ( ! empty( $result['site_id'] ) ) {
                update_option( 'seoauto_site_id', $result['site_id'] );
                update_option( 'seoauto_connected_at', current_time( 'mysql' ) );
            }
            if ( ! empty( $result['website_name'] ) ) {
                update_option( 'seoauto_website_name', $result['website_name'] );
            }
            if ( ! empty( $result['website_url'] ) ) {
                update_option( 'seoauto_website_url', $result['website_url'] );
            }

            // Mark connection as valid
            $plugin->set_connection_status( true );
            $plugin->set_connection_error( null );

            add_settings_error(
                'seoauto_api_key',
                'api_key_valid',
                __( 'API key validated successfully!', 'seoauto-ai-content-publisher' ),
                'success'
            );

        } catch ( \Exception $e ) {
            // Mark connection as invalid
            $plugin->set_connection_status( false );
            $plugin->set_connection_error( $e->getMessage() );

            add_settings_error(
                'seoauto_api_key',
                'api_key_invalid',
                sprintf(
                    /* translators: %s: Error message */
                    __( 'API key validation failed: %s', 'seoauto-ai-content-publisher' ),
                    $e->getMessage()
                ),
                'error'
            );

            // Return old value on failure
            return get_option( 'seoauto_api_key', '' );
        }

        return $value;
    }

    /**
     * AJAX handler for API key validation.
     *
     * @return void
     */
    public function ajax_validate_api_key(): void {
        check_ajax_referer( 'seoauto_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized', 'seoauto-ai-content-publisher' ) ), 403 );
        }

        $api_key = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';
        $plugin  = \SEOAuto\Plugin\Plugin::get_instance();

        if ( empty( $api_key ) ) {
            $plugin->set_connection_status( false );
            $plugin->set_connection_error( __( 'API key is required.', 'seoauto-ai-content-publisher' ) );
            wp_send_json_error( array( 'message' => __( 'API key is required.', 'seoauto-ai-content-publisher' ) ) );
        }

        try {
            $auth   = new Authentication();
            $result = $auth->validate_api_key( $api_key );

            // Update connection status to valid
            $plugin->set_connection_status( true );
            $plugin->set_connection_error( null );

            // Store site info
            if ( ! empty( $result['site_id'] ) ) {
                update_option( 'seoauto_site_id', $result['site_id'] );
                update_option( 'seoauto_connected_at', current_time( 'mysql' ) );
            }
            if ( ! empty( $result['website_name'] ) ) {
                update_option( 'seoauto_website_name', $result['website_name'] );
            }
            if ( ! empty( $result['website_url'] ) ) {
                update_option( 'seoauto_website_url', $result['website_url'] );
            }

            wp_send_json_success(
                array(
                    'message'      => __( 'API key is valid!', 'seoauto-ai-content-publisher' ),
                    'site_id'      => $result['site_id'] ?? '',
                    'website_name' => $result['website_name'] ?? '',
                    'website_url'  => $result['website_url'] ?? '',
                    'plan'         => $result['plan'] ?? '',
                    'connected'    => true,
                )
            );

        } catch ( \Exception $e ) {
            // Mark connection as invalid
            $plugin->set_connection_status( false );
            $plugin->set_connection_error( $e->getMessage() );

            wp_send_json_error(
                array(
                    'message'   => $e->getMessage(),
                    'connected' => false,
                )
            );
        }
    }

    /**
     * AJAX handler for connection test.
     *
     * @return void
     */
    public function ajax_test_connection(): void {
        check_ajax_referer( 'seoauto_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized', 'seoauto-ai-content-publisher' ) ), 403 );
        }

        $api_key = get_option( 'seoauto_api_key', '' );
        $plugin  = \SEOAuto\Plugin\Plugin::get_instance();

        if ( empty( $api_key ) ) {
            $plugin->set_connection_status( false );
            $plugin->set_connection_error( __( 'No API key configured.', 'seoauto-ai-content-publisher' ) );
            wp_send_json_error(
                array(
                    'message'   => __( 'No API key configured.', 'seoauto-ai-content-publisher' ),
                    'connected' => false,
                )
            );
        }

        try {
            $auth   = new Authentication();
            $result = $auth->validate_api_key( $api_key );

            // Update connection status to valid
            $plugin->set_connection_status( true );
            $plugin->set_connection_error( null );

            // Update site info from response
            if ( ! empty( $result['site_id'] ) ) {
                update_option( 'seoauto_site_id', $result['site_id'] );
            }
            if ( ! empty( $result['website_name'] ) ) {
                update_option( 'seoauto_website_name', $result['website_name'] );
            }
            if ( ! empty( $result['website_url'] ) ) {
                update_option( 'seoauto_website_url', $result['website_url'] );
            }

            wp_send_json_success(
                array(
                    'message'      => __( 'Connection successful!', 'seoauto-ai-content-publisher' ),
                    'connected'    => true,
                    'website_name' => $result['website_name'] ?? '',
                    'website_url'  => $result['website_url'] ?? '',
                )
            );

        } catch ( \Exception $e ) {
            // Mark connection as invalid
            $plugin->set_connection_status( false );
            $plugin->set_connection_error( $e->getMessage() );

            wp_send_json_error(
                array(
                    'message'   => $e->getMessage(),
                    'connected' => false,
                )
            );
        }
    }

    /**
     * AJAX handler for resetting plugin.
     *
     * @return void
     */
    public function ajax_reset_plugin(): void {
        check_ajax_referer( 'seoauto_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized', 'seoauto-ai-content-publisher' ) ), 403 );
        }

        // Get all seoauto options
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup operation.
        $wpdb->query(
            "DELETE FROM {$wpdb->options}
            WHERE option_name LIKE 'seoauto_%'"
        );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup operation.
        $wpdb->query(
            "DELETE FROM {$wpdb->options}
            WHERE option_name LIKE '_transient_seoauto_%'
            OR option_name LIKE '_transient_timeout_seoauto_%'"
        );

        // Clear scheduled hooks
        wp_clear_scheduled_hook( 'seoauto_auto_sync' );
        wp_clear_scheduled_hook( 'seoauto_cleanup' );
        wp_clear_scheduled_hook( 'seoauto_health_check' );

        // Re-run activation to set defaults
        \SEOAuto\Plugin\Activator::activate();

        wp_send_json_success( array( 'message' => __( 'Plugin reset successfully!', 'seoauto-ai-content-publisher' ) ) );
    }

    /**
     * Get option group name.
     *
     * @return string
     */
    public static function get_option_group(): string {
        return self::OPTION_GROUP;
    }
}
