<?php
/**
 * Admin Notices Class
 *
 * @package SEOAuto\Plugin\Admin
 */

namespace SEOAuto\Plugin\Admin;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles admin notices and notifications.
 *
 * @since 1.0.0
 */
class Notices {

    /**
     * Constructor.
     */
    public function __construct() {
        add_action( 'admin_notices', array( $this, 'display_notices' ) );
        add_action( 'wp_ajax_seoauto_dismiss_notice', array( $this, 'ajax_dismiss_notice' ) );
        add_action( 'wp_ajax_seoauto_manual_sync', array( $this, 'ajax_manual_sync' ) );
    }

    /**
     * Display admin notices.
     *
     * @return void
     */
    public function display_notices(): void {
        // Only show on SEOAuto pages or dashboard
        $screen = get_current_screen();

        if ( ! $screen || ( strpos( $screen->id, 'seoauto-ai-content-publisher' ) === false && 'dashboard' !== $screen->id ) ) {
            return;
        }

        $this->maybe_show_connection_notice();
        $this->maybe_show_sync_warning();
        $this->maybe_show_cron_warning();
    }

    /**
     * Show connection notice if not connected.
     *
     * @return void
     */
    private function maybe_show_connection_notice(): void {
        $api_key = get_option( 'seoauto_api_key', '' );

        if ( empty( $api_key ) ) {
            ?>
            <div class="notice notice-warning is-dismissible seoauto-notice" data-notice="connection">
                <p>
                    <strong><?php esc_html_e( 'SEOAuto', 'seoauto-ai-content-publisher' ); ?>:</strong>
                    <?php esc_html_e( 'Please configure your API key to start receiving articles.', 'seoauto-ai-content-publisher' ); ?>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=seoauto-settings' ) ); ?>">
                        <?php esc_html_e( 'Go to Settings', 'seoauto-ai-content-publisher' ); ?>
                    </a>
                </p>
            </div>
            <?php
        }
    }

    /**
     * Show warning if sync hasn't run in 24 hours.
     *
     * @return void
     */
    private function maybe_show_sync_warning(): void {
        // Check if dismissed recently
        if ( get_transient( 'seoauto_sync_warning_dismissed' ) ) {
            return;
        }

        $last_sync = get_option( 'seoauto_last_sync', '' );
        $api_key   = get_option( 'seoauto_api_key', '' );

        // Only show if connected
        if ( empty( $api_key ) ) {
            return;
        }

        // Only show if last sync was more than 24 hours ago
        if ( ! empty( $last_sync ) ) {
            $last_sync_time = strtotime( $last_sync );
            $hours_ago      = ( time() - $last_sync_time ) / 3600;

            if ( $hours_ago < 24 ) {
                return;
            }

            $time_ago = human_time_diff( $last_sync_time, current_time( 'timestamp' ) );
        } else {
            $time_ago = __( 'never', 'seoauto-ai-content-publisher' );
        }

        ?>
        <div class="notice notice-warning is-dismissible seoauto-notice" data-notice="sync_warning">
            <p>
                <strong><?php esc_html_e( 'SEOAuto', 'seoauto-ai-content-publisher' ); ?>:</strong>
                <?php
                printf(
                    /* translators: %s: Time since last sync */
                    esc_html__( 'Last sync was %s ago. Consider running a manual sync.', 'seoauto-ai-content-publisher' ),
                    esc_html( $time_ago )
                );
                ?>
                <button type="button" class="button button-small seoauto-sync-now">
                    <?php esc_html_e( 'Sync Now', 'seoauto-ai-content-publisher' ); ?>
                </button>
            </p>
        </div>
        <?php
    }

    /**
     * Show warning if WordPress cron is disabled.
     *
     * @return void
     */
    private function maybe_show_cron_warning(): void {
        // Check if dismissed recently
        if ( get_transient( 'seoauto_cron_warning_dismissed' ) ) {
            return;
        }

        if ( defined( 'DISABLE_WP_CRON' ) && DISABLE_WP_CRON ) {
            ?>
            <div class="notice notice-warning is-dismissible seoauto-notice" data-notice="cron_warning">
                <p>
                    <strong><?php esc_html_e( 'SEOAuto', 'seoauto-ai-content-publisher' ); ?>:</strong>
                    <?php esc_html_e( 'WordPress cron is disabled. Automatic article sync may not work. Please set up a real cron job or enable WP-Cron.', 'seoauto-ai-content-publisher' ); ?>
                    <a href="https://developer.wordpress.org/plugins/cron/" target="_blank" rel="noopener">
                        <?php esc_html_e( 'Learn more', 'seoauto-ai-content-publisher' ); ?>
                    </a>
                </p>
            </div>
            <?php
        }
    }

    /**
     * AJAX handler for dismissing notices.
     *
     * @return void
     */
    public function ajax_dismiss_notice(): void {
        check_ajax_referer( 'seoauto_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized', 'seoauto-ai-content-publisher' ) ), 403 );
        }

        $notice = isset( $_POST['notice'] ) ? sanitize_text_field( wp_unslash( $_POST['notice'] ) ) : '';

        if ( empty( $notice ) ) {
            wp_send_json_error( array( 'message' => __( 'Invalid notice', 'seoauto-ai-content-publisher' ) ) );
        }

        // Set transient to dismiss for 1 hour
        set_transient( 'seoauto_' . $notice . '_dismissed', true, HOUR_IN_SECONDS );

        wp_send_json_success( array( 'message' => __( 'Notice dismissed', 'seoauto-ai-content-publisher' ) ) );
    }

    /**
     * AJAX handler for manual sync.
     *
     * @return void
     */
    public function ajax_manual_sync(): void {
        check_ajax_referer( 'seoauto_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized', 'seoauto-ai-content-publisher' ) ), 403 );
        }

        // Trigger sync action
        do_action( 'seoauto_sync_started' );

        try {
            // This will be implemented in the Sync classes
            $sync_manager = new \SEOAuto\Plugin\Sync\SyncManager();
            $result       = $sync_manager->sync();

            update_option( 'seoauto_last_sync', current_time( 'mysql' ) );

            do_action( 'seoauto_sync_completed', $result );

            wp_send_json_success(
                array(
                    'message'       => __( 'Sync completed successfully!', 'seoauto-ai-content-publisher' ),
                    'articles_new'  => $result['new'] ?? 0,
                    'articles_updated' => $result['updated'] ?? 0,
                )
            );

        } catch ( \Exception $e ) {
            do_action( 'seoauto_sync_failed', $e );

            wp_send_json_error( array( 'message' => $e->getMessage() ) );
        }
    }
}
