<?php
/**
 * Articles List Table Class
 *
 * @package SEOAuto\Plugin\Admin
 */

namespace SEOAuto\Plugin\Admin;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load WP_List_Table if not available
if ( ! class_exists( 'WP_List_Table' ) ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Extends WP_List_Table for SEOAuto articles.
 *
 * @since 1.0.0
 */
class ArticlesTable extends \WP_List_Table {

    /**
     * Table name.
     *
     * @var string
     */
    private string $table_name;

    /**
     * Constructor.
     */
    public function __construct() {
        global $wpdb;

        $this->table_name = $wpdb->prefix . 'seoauto_articles';

        parent::__construct(
            array(
                'singular' => __( 'Article', 'seoauto-ai-content-publisher' ),
                'plural'   => __( 'Articles', 'seoauto-ai-content-publisher' ),
                'ajax'     => false,
            )
        );
    }

    /**
     * Get columns.
     *
     * @return array
     */
    public function get_columns(): array {
        return array(
            'cb'           => '<input type="checkbox" />',
            'title'        => __( 'Title', 'seoauto-ai-content-publisher' ),
            'status'       => __( 'Status', 'seoauto-ai-content-publisher' ),
            'seoauto_id'  => __( 'SEOAuto ID', 'seoauto-ai-content-publisher' ),
            'synced_at'    => __( 'Synced', 'seoauto-ai-content-publisher' ),
            'published_at' => __( 'Published', 'seoauto-ai-content-publisher' ),
        );
    }

    /**
     * Get sortable columns.
     *
     * @return array
     */
    protected function get_sortable_columns(): array {
        return array(
            'title'        => array( 'title', false ),
            'status'       => array( 'status', false ),
            'synced_at'    => array( 'synced_at', true ),
            'published_at' => array( 'published_at', false ),
        );
    }

    /**
     * Get bulk actions.
     *
     * @return array
     */
    protected function get_bulk_actions(): array {
        return array(
            'republish' => __( 'Republish', 'seoauto-ai-content-publisher' ),
            'delete'    => __( 'Delete', 'seoauto-ai-content-publisher' ),
        );
    }

    /**
     * Checkbox column.
     *
     * @param array $item Row item.
     * @return string
     */
    protected function column_cb( $item ): string {
        return sprintf(
            '<input type="checkbox" name="article_ids[]" value="%s" />',
            esc_attr( $item['id'] )
        );
    }

    /**
     * Title column.
     *
     * @param array $item Row item.
     * @return string
     */
    protected function column_title( $item ): string {
        $title = ! empty( $item['title'] ) ? $item['title'] : __( '(No title)', 'seoauto-ai-content-publisher' );

        // Build row actions
        $actions = array();

        if ( ! empty( $item['post_id'] ) ) {
            $actions['view'] = sprintf(
                '<a href="%s" target="_blank">%s</a>',
                esc_url( get_permalink( $item['post_id'] ) ),
                __( 'View', 'seoauto-ai-content-publisher' )
            );

            $actions['edit'] = sprintf(
                '<a href="%s">%s</a>',
                esc_url( get_edit_post_link( $item['post_id'] ) ),
                __( 'Edit', 'seoauto-ai-content-publisher' )
            );
        }

        $actions['delete'] = sprintf(
            '<a href="%s" class="seoauto-delete-article" data-id="%s">%s</a>',
            wp_nonce_url(
                admin_url( 'admin.php?page=seoauto-articles&action=delete&article_id=' . $item['id'] ),
                'seoauto_delete_article'
            ),
            esc_attr( $item['id'] ),
            __( 'Delete', 'seoauto-ai-content-publisher' )
        );

        // Build title link
        $title_output = $title;
        if ( ! empty( $item['post_id'] ) ) {
            $title_output = sprintf(
                '<a href="%s"><strong>%s</strong></a>',
                esc_url( get_edit_post_link( $item['post_id'] ) ),
                esc_html( $title )
            );
        } else {
            $title_output = '<strong>' . esc_html( $title ) . '</strong>';
        }

        return $title_output . $this->row_actions( $actions );
    }

    /**
     * Status column.
     *
     * @param array $item Row item.
     * @return string
     */
    protected function column_status( $item ): string {
        $status = $item['status'] ?? 'pending';

        $status_labels = array(
            'pending'   => __( 'Pending', 'seoauto-ai-content-publisher' ),
            'published' => __( 'Published', 'seoauto-ai-content-publisher' ),
            'failed'    => __( 'Failed', 'seoauto-ai-content-publisher' ),
            'deleted'   => __( 'Deleted', 'seoauto-ai-content-publisher' ),
        );

        $label = $status_labels[ $status ] ?? ucfirst( $status );

        return sprintf(
            '<span class="seoauto-status-badge %s">%s</span>',
            esc_attr( $status ),
            esc_html( $label )
        );
    }

    /**
     * SEOAuto ID column.
     *
     * @param array $item Row item.
     * @return string
     */
    protected function column_seoauto_id( $item ): string {
        return '<code>' . esc_html( $item['seoauto_id'] ) . '</code>';
    }

    /**
     * Synced at column.
     *
     * @param array $item Row item.
     * @return string
     */
    protected function column_synced_at( $item ): string {
        if ( empty( $item['synced_at'] ) ) {
            return '&mdash;';
        }

        $timestamp = strtotime( $item['synced_at'] );

        return sprintf(
            '<abbr title="%s">%s</abbr>',
            esc_attr( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp ) ),
            esc_html( human_time_diff( $timestamp, time() ) . ' ' . __( 'ago', 'seoauto-ai-content-publisher' ) )
        );
    }

    /**
     * Published at column.
     *
     * @param array $item Row item.
     * @return string
     */
    protected function column_published_at( $item ): string {
        if ( empty( $item['published_at'] ) ) {
            return '&mdash;';
        }

        $timestamp = strtotime( $item['published_at'] );

        return sprintf(
            '<abbr title="%s">%s</abbr>',
            esc_attr( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp ) ),
            esc_html( human_time_diff( $timestamp, time() ) . ' ' . __( 'ago', 'seoauto-ai-content-publisher' ) )
        );
    }

    /**
     * Default column handler.
     *
     * @param array  $item        Row item.
     * @param string $column_name Column name.
     * @return string
     */
    protected function column_default( $item, $column_name ): string {
        return isset( $item[ $column_name ] ) ? esc_html( $item[ $column_name ] ) : '';
    }

    /**
     * Get views (status filters).
     *
     * @return array
     */
    protected function get_views(): array {
        global $wpdb;

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only filter for view highlighting.
        $current_status = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : '';
        $base_url       = admin_url( 'admin.php?page=seoauto-articles' );

        // Get counts.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query.
        $counts = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
            "SELECT status, COUNT(*) as count FROM {$this->table_name} GROUP BY status",
            ARRAY_A
        );

        $status_counts = array(
            'all'       => 0,
            'pending'   => 0,
            'published' => 0,
            'failed'    => 0,
            'deleted'   => 0,
        );

        foreach ( $counts as $row ) {
            if ( isset( $status_counts[ $row['status'] ] ) ) {
                $status_counts[ $row['status'] ] = (int) $row['count'];
            }
            $status_counts['all'] += (int) $row['count'];
        }

        $views = array();

        // All
        $views['all'] = sprintf(
            '<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
            esc_url( $base_url ),
            empty( $current_status ) ? 'current' : '',
            __( 'All', 'seoauto-ai-content-publisher' ),
            $status_counts['all']
        );

        // Published
        if ( $status_counts['published'] > 0 ) {
            $views['published'] = sprintf(
                '<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
                esc_url( add_query_arg( 'status', 'published', $base_url ) ),
                'published' === $current_status ? 'current' : '',
                __( 'Published', 'seoauto-ai-content-publisher' ),
                $status_counts['published']
            );
        }

        // Pending
        if ( $status_counts['pending'] > 0 ) {
            $views['pending'] = sprintf(
                '<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
                esc_url( add_query_arg( 'status', 'pending', $base_url ) ),
                'pending' === $current_status ? 'current' : '',
                __( 'Pending', 'seoauto-ai-content-publisher' ),
                $status_counts['pending']
            );
        }

        // Failed
        if ( $status_counts['failed'] > 0 ) {
            $views['failed'] = sprintf(
                '<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
                esc_url( add_query_arg( 'status', 'failed', $base_url ) ),
                'failed' === $current_status ? 'current' : '',
                __( 'Failed', 'seoauto-ai-content-publisher' ),
                $status_counts['failed']
            );
        }

        return $views;
    }

    /**
     * Prepare items for display.
     *
     * @return void
     */
    public function prepare_items(): void {
        global $wpdb;

        $per_page = 20;
        $columns  = $this->get_columns();
        $hidden   = array();
        $sortable = $this->get_sortable_columns();

        $this->_column_headers = array( $columns, $hidden, $sortable );

        // Process bulk actions
        $this->process_bulk_action();

        // Get data
        $current_page = $this->get_pagenum();
        $offset       = ( $current_page - 1 ) * $per_page;

        // Build WHERE clause components.
        $status_condition = '';
        $search_condition = '';
        $status_value     = '';
        $search_value     = '';

        // Status filter - whitelist validation to prevent SQL injection.
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only filter.
        if ( ! empty( $_GET['status'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only filter.
            $status_input = sanitize_key( wp_unslash( $_GET['status'] ) );
            // Use switch for static analysis - each case assigns a literal string.
            switch ( $status_input ) {
                case 'pending':
                    $status_value     = 'pending';
                    $status_condition = 'status = %s';
                    break;
                case 'published':
                    $status_value     = 'published';
                    $status_condition = 'status = %s';
                    break;
                case 'failed':
                    $status_value     = 'failed';
                    $status_condition = 'status = %s';
                    break;
                case 'deleted':
                    $status_value     = 'deleted';
                    $status_condition = 'status = %s';
                    break;
            }
        }

        // Search.
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only filter.
        if ( ! empty( $_GET['s'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only filter.
            $search_value     = '%' . $wpdb->esc_like( sanitize_text_field( wp_unslash( $_GET['s'] ) ) ) . '%';
            $search_condition = '(title LIKE %s OR seoauto_id LIKE %s)';
        }

        // Sorting - use switch for static analysis traceability.
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only sort parameter.
        $orderby_input = isset( $_GET['orderby'] ) && is_string( $_GET['orderby'] ) ? sanitize_key( wp_unslash( $_GET['orderby'] ) ) : '';

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only sort parameter.
        $order_input = isset( $_GET['order'] ) && is_string( $_GET['order'] ) ? strtolower( sanitize_key( wp_unslash( $_GET['order'] ) ) ) : '';
        $order_dir   = 'asc' === $order_input ? 'ASC' : 'DESC';

        // Execute query based on filters - using fully static SQL strings.
        $this->items = $this->execute_query( $status_condition, $status_value, $search_condition, $search_value, $orderby_input, $order_dir, $per_page, $offset );

        // Get total count.
        $total_items = $this->get_total_count( $status_condition, $status_value, $search_condition, $search_value );

        // Set pagination
        $this->set_pagination_args(
            array(
                'total_items' => $total_items,
                'per_page'    => $per_page,
                'total_pages' => ceil( $total_items / $per_page ),
            )
        );
    }

    /**
     * Execute the main query with static ORDER BY clauses.
     *
     * Uses switch statement to ensure static analysis can trace the ORDER BY column.
     *
     * @param string $status_condition Status WHERE condition.
     * @param string $status_value     Status value.
     * @param string $search_condition Search WHERE condition.
     * @param string $search_value     Search value.
     * @param string $orderby_input    Order by column input.
     * @param string $order_dir        Order direction (ASC/DESC).
     * @param int    $per_page         Items per page.
     * @param int    $offset           Query offset.
     * @return array Query results.
     */
    private function execute_query( string $status_condition, string $status_value, string $search_condition, string $search_value, string $orderby_input, string $order_dir, int $per_page, int $offset ): array {
        global $wpdb;

        $table = $this->table_name;

        // Build base query parts.
        $has_status = ! empty( $status_condition );
        $has_search = ! empty( $search_condition );

        // Determine ORDER BY using switch - each case uses literal SQL.
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        switch ( $orderby_input ) {
            case 'title':
                if ( $has_status && $has_search ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s) ORDER BY title DESC LIMIT %d OFFSET %d", $status_value, $search_value, $search_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s) ORDER BY title ASC LIMIT %d OFFSET %d", $status_value, $search_value, $search_value, $per_page, $offset ), ARRAY_A );
                } elseif ( $has_status ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s ORDER BY title DESC LIMIT %d OFFSET %d", $status_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s ORDER BY title ASC LIMIT %d OFFSET %d", $status_value, $per_page, $offset ), ARRAY_A );
                } elseif ( $has_search ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s) ORDER BY title DESC LIMIT %d OFFSET %d", $search_value, $search_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s) ORDER BY title ASC LIMIT %d OFFSET %d", $search_value, $search_value, $per_page, $offset ), ARRAY_A );
                } else {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY title DESC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY title ASC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A );
                }

            case 'status':
                if ( $has_status && $has_search ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s) ORDER BY status DESC LIMIT %d OFFSET %d", $status_value, $search_value, $search_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s) ORDER BY status ASC LIMIT %d OFFSET %d", $status_value, $search_value, $search_value, $per_page, $offset ), ARRAY_A );
                } elseif ( $has_status ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s ORDER BY status DESC LIMIT %d OFFSET %d", $status_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s ORDER BY status ASC LIMIT %d OFFSET %d", $status_value, $per_page, $offset ), ARRAY_A );
                } elseif ( $has_search ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s) ORDER BY status DESC LIMIT %d OFFSET %d", $search_value, $search_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s) ORDER BY status ASC LIMIT %d OFFSET %d", $search_value, $search_value, $per_page, $offset ), ARRAY_A );
                } else {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY status DESC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY status ASC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A );
                }

            case 'published_at':
                if ( $has_status && $has_search ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s) ORDER BY published_at DESC LIMIT %d OFFSET %d", $status_value, $search_value, $search_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s) ORDER BY published_at ASC LIMIT %d OFFSET %d", $status_value, $search_value, $search_value, $per_page, $offset ), ARRAY_A );
                } elseif ( $has_status ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s ORDER BY published_at DESC LIMIT %d OFFSET %d", $status_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s ORDER BY published_at ASC LIMIT %d OFFSET %d", $status_value, $per_page, $offset ), ARRAY_A );
                } elseif ( $has_search ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s) ORDER BY published_at DESC LIMIT %d OFFSET %d", $search_value, $search_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s) ORDER BY published_at ASC LIMIT %d OFFSET %d", $search_value, $search_value, $per_page, $offset ), ARRAY_A );
                } else {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY published_at DESC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY published_at ASC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A );
                }

            case 'seoauto_id':
                if ( $has_status && $has_search ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s) ORDER BY seoauto_id DESC LIMIT %d OFFSET %d", $status_value, $search_value, $search_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s) ORDER BY seoauto_id ASC LIMIT %d OFFSET %d", $status_value, $search_value, $search_value, $per_page, $offset ), ARRAY_A );
                } elseif ( $has_status ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s ORDER BY seoauto_id DESC LIMIT %d OFFSET %d", $status_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s ORDER BY seoauto_id ASC LIMIT %d OFFSET %d", $status_value, $per_page, $offset ), ARRAY_A );
                } elseif ( $has_search ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s) ORDER BY seoauto_id DESC LIMIT %d OFFSET %d", $search_value, $search_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s) ORDER BY seoauto_id ASC LIMIT %d OFFSET %d", $search_value, $search_value, $per_page, $offset ), ARRAY_A );
                } else {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY seoauto_id DESC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY seoauto_id ASC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A );
                }

            case 'synced_at':
            default:
                // Default sort by synced_at.
                if ( $has_status && $has_search ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s) ORDER BY synced_at DESC LIMIT %d OFFSET %d", $status_value, $search_value, $search_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s) ORDER BY synced_at ASC LIMIT %d OFFSET %d", $status_value, $search_value, $search_value, $per_page, $offset ), ARRAY_A );
                } elseif ( $has_status ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s ORDER BY synced_at DESC LIMIT %d OFFSET %d", $status_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE status = %s ORDER BY synced_at ASC LIMIT %d OFFSET %d", $status_value, $per_page, $offset ), ARRAY_A );
                } elseif ( $has_search ) {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s) ORDER BY synced_at DESC LIMIT %d OFFSET %d", $search_value, $search_value, $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s) ORDER BY synced_at ASC LIMIT %d OFFSET %d", $search_value, $search_value, $per_page, $offset ), ARRAY_A );
                } else {
                    return 'DESC' === $order_dir
                        ? $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY synced_at DESC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A )
                        : $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY synced_at ASC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A );
                }
        }
        // phpcs:enable
    }

    /**
     * Get total count for pagination.
     *
     * @param string $status_condition Status WHERE condition.
     * @param string $status_value     Status value.
     * @param string $search_condition Search WHERE condition.
     * @param string $search_value     Search value.
     * @return int Total count.
     */
    private function get_total_count( string $status_condition, string $status_value, string $search_condition, string $search_value ): int {
        global $wpdb;

        $table      = $this->table_name;
        $has_status = ! empty( $status_condition );
        $has_search = ! empty( $search_condition );

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        if ( $has_status && $has_search ) {
            return (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table} WHERE status = %s AND (title LIKE %s OR seoauto_id LIKE %s)", $status_value, $search_value, $search_value ) );
        } elseif ( $has_status ) {
            return (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table} WHERE status = %s", $status_value ) );
        } elseif ( $has_search ) {
            return (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table} WHERE (title LIKE %s OR seoauto_id LIKE %s)", $search_value, $search_value ) );
        } else {
            return (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );
        }
        // phpcs:enable
    }

    /**
     * Process bulk actions.
     *
     * @return void
     */
    public function process_bulk_action(): void {
        global $wpdb;

        // Single delete action
        if ( 'delete' === $this->current_action() && ! empty( $_GET['article_id'] ) ) {
            check_admin_referer( 'seoauto_delete_article' );

            $article_id = absint( $_GET['article_id'] );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table.
            $article    = $wpdb->get_row(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
                    "SELECT * FROM {$this->table_name} WHERE id = %d",
                    $article_id
                ),
                ARRAY_A
            );

            if ( $article ) {
                // Trash the post if exists
                if ( ! empty( $article['post_id'] ) ) {
                    wp_trash_post( $article['post_id'] );
                }

                // Delete the record
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table delete.
                $wpdb->delete( $this->table_name, array( 'id' => $article_id ), array( '%d' ) );
            }

            wp_safe_redirect( admin_url( 'admin.php?page=seoauto-articles&deleted=1' ) );
            exit;
        }

        // Bulk actions
        if ( ! empty( $_POST['article_ids'] ) && is_array( $_POST['article_ids'] ) ) {
            check_admin_referer( 'bulk-articles' );

            $article_ids = array_map( 'absint', $_POST['article_ids'] );

            if ( 'delete' === $this->current_action() ) {
                foreach ( $article_ids as $article_id ) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table.
                    $article = $wpdb->get_row(
                        $wpdb->prepare(
                            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix.
                            "SELECT * FROM {$this->table_name} WHERE id = %d",
                            $article_id
                        ),
                        ARRAY_A
                    );

                    if ( $article ) {
                        if ( ! empty( $article['post_id'] ) ) {
                            wp_trash_post( $article['post_id'] );
                        }
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Delete operation.
                        $wpdb->delete( $this->table_name, array( 'id' => $article_id ), array( '%d' ) );
                    }
                }

                wp_safe_redirect( admin_url( 'admin.php?page=seoauto-articles&deleted=' . count( $article_ids ) ) );
                exit;
            }
        }
    }

    /**
     * Display when no items.
     *
     * @return void
     */
    public function no_items(): void {
        esc_html_e( 'No articles found.', 'seoauto-ai-content-publisher' );
    }

    /**
     * Extra table nav (search box).
     *
     * @param string $which Position (top or bottom).
     * @return void
     */
    protected function extra_tablenav( $which ): void {
        if ( 'top' === $which ) {
            ?>
            <div class="alignleft actions">
                <?php
                // Add any additional filters here
                ?>
            </div>
            <?php
        }
    }
}
