<?php
/**
 * Admin Menu Class
 *
 * @package SEOAuto\Plugin\Admin
 */

namespace SEOAuto\Plugin\Admin;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles admin menu registration.
 *
 * @since 1.0.0
 */
class AdminMenu {

    /**
     * Menu capability required.
     *
     * @var string
     */
    private string $capability = 'manage_options';

    /**
     * Menu icon.
     *
     * @var string
     */
    private string $icon = '';

    /**
     * Register admin menus.
     *
     * @return void
     */
    public function register(): void {
        /**
         * Filter the capability required to access SEOAuto admin.
         *
         * @param string $capability Default capability.
         */
        $this->capability = apply_filters( 'seoauto_menu_capability', $this->capability );

        // Use custom icon or fallback to dashicon
        $icon_url = SEOAUTO_PLUGIN_URL . 'assets/images/menu-icon.png';
        if ( ! file_exists( SEOAUTO_PLUGIN_DIR . 'assets/images/menu-icon.png' ) ) {
            $icon_url = 'dashicons-chart-line';
        }

        // Main menu - Dashboard
        add_menu_page(
            __( 'SEOAuto', 'seoauto-ai-content-publisher' ),
            __( 'SEOAuto', 'seoauto-ai-content-publisher' ),
            $this->capability,
            'seoauto-ai-content-publisher',
            array( $this, 'render_dashboard_page' ),
            $icon_url,
            30
        );

        // Dashboard submenu (same as parent)
        add_submenu_page(
            'seoauto-ai-content-publisher',
            __( 'Dashboard', 'seoauto-ai-content-publisher' ),
            __( 'Dashboard', 'seoauto-ai-content-publisher' ),
            $this->capability,
            'seoauto-ai-content-publisher',
            array( $this, 'render_dashboard_page' )
        );

        // Articles submenu
        add_submenu_page(
            'seoauto-ai-content-publisher',
            __( 'Articles', 'seoauto-ai-content-publisher' ),
            __( 'Articles', 'seoauto-ai-content-publisher' ),
            $this->capability,
            'seoauto-articles',
            array( $this, 'render_articles_page' )
        );

        // Settings submenu
        add_submenu_page(
            'seoauto-ai-content-publisher',
            __( 'Settings', 'seoauto-ai-content-publisher' ),
            __( 'Settings', 'seoauto-ai-content-publisher' ),
            $this->capability,
            'seoauto-settings',
            array( $this, 'render_settings_page' )
        );
    }

    /**
     * Render dashboard page.
     *
     * @return void
     */
    public function render_dashboard_page(): void {
        // Check user capabilities
        if ( ! current_user_can( $this->capability ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'seoauto-ai-content-publisher' ) );
        }

        $template = SEOAUTO_PLUGIN_DIR . 'templates/admin/dashboard.php';

        if ( file_exists( $template ) ) {
            include $template;
        } else {
            $this->render_placeholder( 'Dashboard' );
        }
    }

    /**
     * Render articles page.
     *
     * @return void
     */
    public function render_articles_page(): void {
        // Check user capabilities
        if ( ! current_user_can( $this->capability ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'seoauto-ai-content-publisher' ) );
        }

        $template = SEOAUTO_PLUGIN_DIR . 'templates/admin/articles.php';

        if ( file_exists( $template ) ) {
            include $template;
        } else {
            $this->render_placeholder( 'Articles' );
        }
    }

    /**
     * Render settings page.
     *
     * @return void
     */
    public function render_settings_page(): void {
        // Check user capabilities
        if ( ! current_user_can( $this->capability ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'seoauto-ai-content-publisher' ) );
        }

        $template = SEOAUTO_PLUGIN_DIR . 'templates/admin/settings.php';

        if ( file_exists( $template ) ) {
            include $template;
        } else {
            $this->render_placeholder( 'Settings' );
        }
    }

    /**
     * Render placeholder for missing templates.
     *
     * @param string $page_name Page name to display.
     * @return void
     */
    private function render_placeholder( string $page_name ): void {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( 'SEOAuto - ' . $page_name ); ?></h1>
            <div class="notice notice-info">
                <p>
                    <?php
                    printf(
                        /* translators: %s: Page name */
                        esc_html__( 'The %s template is being developed.', 'seoauto-ai-content-publisher' ),
                        esc_html( $page_name )
                    );
                    ?>
                </p>
            </div>
        </div>
        <?php
    }
}
