<?php
/**
 * Plugin Activator
 *
 * @package SEOAuto\Plugin
 */

namespace SEOAuto\Plugin;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SEOAuto\Plugin\Database\Migrator;
use SEOAuto\Plugin\Scheduler\CronManager;

/**
 * Handles plugin activation.
 *
 * @since 1.0.0
 */
class Activator {

    /**
     * Run activation routines.
     *
     * @return void
     */
    public static function activate(): void {
        // Create database tables
        self::create_tables();

        // Set default options
        self::set_default_options();

        // Schedule cron events
        self::schedule_cron_events();

        // Store activation time
        update_option( 'seoauto_activation_time', current_time( 'mysql' ) );

        // Set DB version
        update_option( 'seoauto_db_version', SEOAUTO_VERSION );

        // Flush rewrite rules for REST API
        flush_rewrite_rules();

        // Fire activation action
        do_action( 'seoauto_activated' );
    }

    /**
     * Create database tables.
     *
     * @return void
     */
    private static function create_tables(): void {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        // Articles tracking table (as per PRD section 5.1)
        $articles_table = $wpdb->prefix . 'seoauto_articles';
        $articles_sql   = "CREATE TABLE $articles_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            seoauto_id varchar(100) NOT NULL,
            post_id bigint(20) unsigned DEFAULT NULL,
            title text NOT NULL,
            status enum('pending','published','failed','deleted') DEFAULT 'pending',
            error_message text DEFAULT NULL,
            synced_at datetime NOT NULL,
            published_at datetime DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY seoauto_id (seoauto_id),
            KEY post_id (post_id),
            KEY status (status),
            KEY synced_at (synced_at)
        ) $charset_collate;";

        // Activity logs table (as per PRD section 5.1)
        $logs_table = $wpdb->prefix . 'seoauto_logs';
        $logs_sql   = "CREATE TABLE $logs_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            level enum('debug','info','warning','error') DEFAULT 'info',
            message text NOT NULL,
            context longtext DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY level (level),
            KEY created_at (created_at)
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        dbDelta( $articles_sql );
        dbDelta( $logs_sql );
    }

    /**
     * Set default plugin options.
     *
     * @return void
     */
    private static function set_default_options(): void {
        // Connection (will be set when user connects)
        add_option( 'seoauto_api_key', '' );
        add_option( 'seoauto_site_id', '' );
        add_option( 'seoauto_connected_at', '' );

        // Publishing settings
        add_option( 'seoauto_default_category', 1 );
        add_option( 'seoauto_default_author', get_current_user_id() ?: 1 );
        add_option( 'seoauto_post_status', 'publish' );
        add_option( 'seoauto_auto_publish', true );

        // Content settings
        add_option( 'seoauto_include_youtube', true );
        add_option( 'seoauto_include_infographic', true );
        add_option( 'seoauto_infographic_position', 'end' );

        // Advanced settings
        add_option( 'seoauto_debug_mode', false );
        add_option( 'seoauto_sync_interval', 'hourly' );
        add_option( 'seoauto_last_sync', '' );

        // Internal
        add_option( 'seoauto_setup_complete', false );
    }

    /**
     * Schedule cron events.
     *
     * @return void
     */
    private static function schedule_cron_events(): void {
        // Schedule auto sync based on settings
        $sync_interval = get_option( 'seoauto_sync_interval', 'hourly' );

        if ( ! wp_next_scheduled( 'seoauto_auto_sync' ) ) {
            wp_schedule_event( time(), $sync_interval, 'seoauto_auto_sync' );
        }

        // Schedule daily cleanup
        if ( ! wp_next_scheduled( 'seoauto_cleanup' ) ) {
            wp_schedule_event( time(), 'daily', 'seoauto_cleanup' );
        }

        // Schedule health check twice daily
        if ( ! wp_next_scheduled( 'seoauto_health_check' ) ) {
            wp_schedule_event( time(), 'twicedaily', 'seoauto_health_check' );
        }
    }

    /**
     * Update database schema (called on plugin update).
     *
     * @return void
     */
    public static function update_database(): void {
        self::create_tables();
    }
}
