<?php
/**
 * SEOAuto AI Content Publisher
 *
 * @package     SEOAuto
 * @author      SEOAuto
 * @copyright   2025 SEOAuto
 * @license     GPL-2.0-or-later
 *
 * @wordpress-plugin
 * Plugin Name:       SEOAuto AI Content Publisher
 * Description:       Automatically publish AI-generated SEO articles from SEOAuto to your WordPress site.
 * Version:           1.2.1
 * Requires at least: 6.0
 * Requires PHP:      7.4
 * Author:            SEOAuto
 * Author URI:        https://seoauto.io
 * Text Domain:       seoauto-ai-content-publisher
 * Domain Path:       /languages
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Plugin version.
 */
define( 'SEOAUTO_VERSION', '1.2.1' );

/**
 * Plugin file path.
 */
define( 'SEOAUTO_PLUGIN_FILE', __FILE__ );

/**
 * Plugin directory path.
 */
define( 'SEOAUTO_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );

/**
 * Plugin directory URL.
 */
define( 'SEOAUTO_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Plugin basename.
 */
define( 'SEOAUTO_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * SEOAuto API base URL.
 * Uses localhost for development, production URL otherwise.
 */
if ( ! defined( 'SEOAUTO_API_BASE_URL' ) ) {
    /**
     * SEOAuto API Base URL Configuration
     *
     * To override for local testing, add to wp-config.php:
     * define( 'SEOAUTO_API_BASE_URL', 'http://127.0.0.1:8000/api' );
     *
     * Or set environment variable SEOAUTO_DEV_MODE=1 to force localhost
     */

    // Check for forced development mode
    $seoauto_force_dev = getenv( 'SEOAUTO_DEV_MODE' ) === '1'
        || ( defined( 'SEOAUTO_DEV_MODE' ) && SEOAUTO_DEV_MODE );

    // Check for local development environment
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Used only for string comparison
    $seoauto_server_name = isset( $_SERVER['SERVER_NAME'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_NAME'] ) ) : '';
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Used only for string comparison
    $seoauto_http_host = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';

    $seoauto_is_local = $seoauto_force_dev
        || (
            ( defined( 'WP_DEBUG' ) && WP_DEBUG )
            && (
                in_array( $seoauto_server_name, array( 'localhost', '127.0.0.1' ), true )
                || strpos( $seoauto_http_host, 'localhost' ) !== false
                || strpos( $seoauto_http_host, '.local' ) !== false
            )
        );

    if ( $seoauto_is_local ) {
        define( 'SEOAUTO_API_BASE_URL', 'http://127.0.0.1:8000/api' );
    } else {
        define( 'SEOAUTO_API_BASE_URL', 'https://seoauto-api-production.up.railway.app/api' );
    }
}

/**
 * Minimum PHP version required.
 */
define( 'SEOAUTO_MIN_PHP_VERSION', '7.4' );

/**
 * Minimum WordPress version required.
 */
define( 'SEOAUTO_MIN_WP_VERSION', '6.0' );

/**
 * Check PHP version requirement.
 *
 * @return bool
 */
function seoauto_check_php_version() {
    return version_compare( PHP_VERSION, SEOAUTO_MIN_PHP_VERSION, '>=' );
}

/**
 * Check WordPress version requirement.
 *
 * @return bool
 */
function seoauto_check_wp_version() {
    return version_compare( get_bloginfo( 'version' ), SEOAUTO_MIN_WP_VERSION, '>=' );
}

/**
 * Display admin notice for PHP version requirement.
 *
 * @return void
 */
function seoauto_php_version_notice() {
    ?>
    <div class="notice notice-error">
        <p>
            <?php
            printf(
                /* translators: 1: Required PHP version, 2: Current PHP version */
                esc_html__( 'SEOAuto requires PHP %1$s or higher. Your current PHP version is %2$s. Please upgrade PHP to use this plugin.', 'seoauto-ai-content-publisher' ),
                esc_html( SEOAUTO_MIN_PHP_VERSION ),
                esc_html( PHP_VERSION )
            );
            ?>
        </p>
    </div>
    <?php
}

/**
 * Display admin notice for WordPress version requirement.
 *
 * @return void
 */
function seoauto_wp_version_notice() {
    ?>
    <div class="notice notice-error">
        <p>
            <?php
            printf(
                /* translators: 1: Required WP version, 2: Current WP version */
                esc_html__( 'SEOAuto requires WordPress %1$s or higher. Your current WordPress version is %2$s. Please upgrade WordPress to use this plugin.', 'seoauto-ai-content-publisher' ),
                esc_html( SEOAUTO_MIN_WP_VERSION ),
                esc_html( get_bloginfo( 'version' ) )
            );
            ?>
        </p>
    </div>
    <?php
}

/**
 * Check requirements and initialize plugin.
 *
 * @return void
 */
function seoauto_init() {
    // Check PHP version
    if ( ! seoauto_check_php_version() ) {
        add_action( 'admin_notices', 'seoauto_php_version_notice' );
        return;
    }

    // Check WordPress version
    if ( ! seoauto_check_wp_version() ) {
        add_action( 'admin_notices', 'seoauto_wp_version_notice' );
        return;
    }

    // Load Composer autoloader
    $autoloader = SEOAUTO_PLUGIN_DIR . 'vendor/autoload.php';
    if ( ! file_exists( $autoloader ) ) {
        add_action( 'admin_notices', 'seoauto_autoloader_notice' );
        return;
    }

    require_once $autoloader;

    // Initialize the plugin
    \SEOAuto\Plugin\Plugin::get_instance();
}

/**
 * Display admin notice when autoloader is missing.
 *
 * @return void
 */
function seoauto_autoloader_notice() {
    ?>
    <div class="notice notice-error">
        <p>
            <?php
            esc_html_e( 'SEOAuto: Composer autoloader not found. Please run "composer install" in the plugin directory.', 'seoauto-ai-content-publisher' );
            ?>
        </p>
    </div>
    <?php
}

/**
 * Plugin activation hook.
 *
 * @return void
 */
function seoauto_activate() {
    // Check requirements first
    if ( ! seoauto_check_php_version() || ! seoauto_check_wp_version() ) {
        deactivate_plugins( plugin_basename( __FILE__ ) );
        wp_die(
            esc_html__( 'SEOAuto requires PHP 7.4+ and WordPress 6.0+.', 'seoauto-ai-content-publisher' ),
            'Plugin Activation Error',
            array( 'back_link' => true )
        );
    }

    // Load autoloader
    $autoloader = SEOAUTO_PLUGIN_DIR . 'vendor/autoload.php';
    if ( file_exists( $autoloader ) ) {
        require_once $autoloader;
        \SEOAuto\Plugin\Activator::activate();
    }
}

/**
 * Plugin deactivation hook.
 *
 * @return void
 */
function seoauto_deactivate() {
    $autoloader = SEOAUTO_PLUGIN_DIR . 'vendor/autoload.php';
    if ( file_exists( $autoloader ) ) {
        require_once $autoloader;
        \SEOAuto\Plugin\Deactivator::deactivate();
    }
}

// Register activation and deactivation hooks
register_activation_hook( __FILE__, 'seoauto_activate' );
register_deactivation_hook( __FILE__, 'seoauto_deactivate' );

// Initialize plugin after WordPress is loaded
add_action( 'plugins_loaded', 'seoauto_init' );
