/**
 * SEOAuto Admin JavaScript
 *
 * @package SEOAuto\Plugin
 */

(function($) {
    'use strict';

    /**
     * SEOAuto Admin Module
     */
    const SEOAutoAdmin = {
        /**
         * Initialize
         */
        init: function() {
            this.bindEvents();
            this.initTabs();
        },

        /**
         * Bind events
         */
        bindEvents: function() {
            // API Key toggle visibility
            $('#seoauto-toggle-key').on('click', this.toggleApiKeyVisibility);

            // Test connection
            $('#seoauto-test-connection').on('click', this.testConnection);

            // Sync now button
            $('#seoauto-sync-now').on('click', this.syncNow);

            // Reset plugin
            $('#seoauto-reset-plugin').on('click', this.resetPlugin);

            // Delete article confirmation
            $('.seoauto-delete-article').on('click', this.confirmDelete);
        },

        /**
         * Initialize tabs on settings page
         */
        initTabs: function() {
            // Check if we're on the settings page with tabs
            if ($('.seoauto-tabs').length === 0) {
                return;
            }

            // Bind click events to tabs
            $('.seoauto-tabs .nav-tab').on('click', function(e) {
                e.preventDefault();

                const tabId = $(this).data('tab');

                // Update tab navigation
                $('.seoauto-tabs .nav-tab').removeClass('nav-tab-active');
                $(this).addClass('nav-tab-active');

                // Show/hide tab content
                $('.seoauto-tab-content').removeClass('active');
                $('#tab-' + tabId).addClass('active');
            });

            // Ensure first tab is visible on load
            if ($('.seoauto-tab-content.active').length === 0) {
                $('.seoauto-tabs .nav-tab').first().addClass('nav-tab-active');
                $('.seoauto-tab-content').first().addClass('active');
            }
        },

        /**
         * Toggle API key visibility
         */
        toggleApiKeyVisibility: function(e) {
            e.preventDefault();

            const $input = $('#seoauto_api_key');
            const $icon = $(this).find('.dashicons');

            if ($input.attr('type') === 'password') {
                $input.attr('type', 'text');
                $icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            } else {
                $input.attr('type', 'password');
                $icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            }
        },

        /**
         * Test connection
         */
        testConnection: function(e) {
            e.preventDefault();

            const $button = $(this);
            const $result = $('#seoauto-validation-result, #seoauto-message');
            const originalHtml = $button.html();
            const apiKey = $('#seoauto_api_key').val();

            // If there's an API key in the field, use it (even if not saved yet)
            const data = {
                action: 'seoauto_test_connection',
                nonce: seoautoAdmin.nonce
            };

            // Include API key if in input field
            if (apiKey) {
                data.action = 'seoauto_validate_api_key';
                data.api_key = apiKey;
            }

            $button.prop('disabled', true).html('<span class="dashicons dashicons-update dashicons-spin"></span> Validating...');

            $.ajax({
                url: seoautoAdmin.ajaxUrl,
                type: 'POST',
                data: data,
                success: function(response) {
                    if (response.success) {
                        SEOAutoAdmin.showMessage($result, 'success', response.data.message);

                        // Show additional info if available
                        if (response.data.website_name) {
                            $result.append('<br><small>Website: ' + response.data.website_name + '</small>');
                        }

                        // Update connection status in UI
                        SEOAutoAdmin.updateConnectionUI(true, response.data);

                        // Reload page after 1.5 seconds to show updated status
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        SEOAutoAdmin.showMessage($result, 'error', response.data.message || seoautoAdmin.strings.error);

                        // Update connection status in UI
                        SEOAutoAdmin.updateConnectionUI(false, response.data);
                    }
                },
                error: function() {
                    SEOAutoAdmin.showMessage($result, 'error', seoautoAdmin.strings.error);
                    SEOAutoAdmin.updateConnectionUI(false, {});
                },
                complete: function() {
                    $button.prop('disabled', false).html(originalHtml);
                }
            });
        },

        /**
         * Update connection status UI dynamically
         */
        updateConnectionUI: function(connected, data) {
            const $connectionBox = $('.seoauto-connection-box');
            const $connectionIcon = $connectionBox.find('.seoauto-connection-icon .dashicons');
            const $connectionDetails = $connectionBox.find('.seoauto-connection-details');

            if (connected) {
                // Update box classes
                $connectionBox.removeClass('disconnected error').addClass('connected');

                // Update icon
                $connectionIcon.removeClass('dashicons-dismiss dashicons-warning').addClass('dashicons-yes-alt');

                // Update text content
                let detailsHtml = '<strong>Connected to SEOAuto</strong>';
                if (data.website_url) {
                    detailsHtml += '<span class="seoauto-connected-site"><span class="dashicons dashicons-admin-site-alt3"></span>' + data.website_url + '</span>';
                } else if (data.website_name) {
                    detailsHtml += '<span class="seoauto-connected-site"><span class="dashicons dashicons-admin-site-alt3"></span>' + data.website_name + '</span>';
                }
                $connectionDetails.html(detailsHtml);
            } else {
                // Update box classes
                $connectionBox.removeClass('connected').addClass('disconnected error');

                // Update icon
                $connectionIcon.removeClass('dashicons-yes-alt dashicons-warning').addClass('dashicons-dismiss');

                // Update text content
                let errorMsg = data.message || 'Connection failed';
                $connectionDetails.html('<strong>Connection Failed</strong><span class="seoauto-error-msg">' + errorMsg + '</span>');
            }

            // Also update dashboard elements if present
            const $statusDot = $('.seoauto-status-dot');
            const $statusText = $('.seoauto-status-text');
            if ($statusDot.length) {
                if (connected) {
                    $statusDot.removeClass('offline error').addClass('online');
                    $statusText.text('Connected to SEOAuto');
                } else {
                    $statusDot.removeClass('online').addClass('offline error');
                    $statusText.text('Connection Failed');
                }
            }
        },

        /**
         * Sync now
         */
        syncNow: function(e) {
            e.preventDefault();

            if (!confirm(seoautoAdmin.strings.confirmSync)) {
                return;
            }

            const $button = $(this);
            const $message = $('#seoauto-sync-message');
            const originalHtml = $button.html();

            $button.prop('disabled', true).html('<span class="dashicons dashicons-update dashicons-spin"></span> Syncing...');

            $.ajax({
                url: seoautoAdmin.restUrl + 'sync',
                type: 'POST',
                headers: {
                    'X-WP-Nonce': seoautoAdmin.restNonce
                },
                success: function(response) {
                    if (response.success) {
                        let msg = seoautoAdmin.strings.syncSuccess;
                        if (response.articles_new > 0 || response.articles_updated > 0) {
                            msg += ' (' + response.articles_new + ' new, ' + response.articles_updated + ' updated)';
                        }
                        SEOAutoAdmin.showMessage($message, 'success', msg);

                        // Reload page after 2 seconds to show updated data
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        SEOAutoAdmin.showMessage($message, 'error', response.message || seoautoAdmin.strings.syncError);
                    }
                },
                error: function(xhr) {
                    let errorMsg = seoautoAdmin.strings.syncError;
                    if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMsg = xhr.responseJSON.message;
                    }
                    SEOAutoAdmin.showMessage($message, 'error', errorMsg);
                },
                complete: function() {
                    $button.prop('disabled', false).html(originalHtml);
                }
            });
        },

        /**
         * Reset plugin
         */
        resetPlugin: function(e) {
            e.preventDefault();

            if (!confirm('Are you sure you want to reset the plugin? All settings will be lost.')) {
                return;
            }

            // Double confirm for safety
            if (!confirm('This action cannot be undone. Are you really sure?')) {
                return;
            }

            const $button = $(this);
            const $message = $('#seoauto-maintenance-message');

            $button.prop('disabled', true);

            $.ajax({
                url: seoautoAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'seoauto_reset_plugin',
                    nonce: seoautoAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        SEOAutoAdmin.showMessage($message, 'success', response.data.message);

                        // Reload page after 2 seconds
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        SEOAutoAdmin.showMessage($message, 'error', response.data.message || seoautoAdmin.strings.error);
                        $button.prop('disabled', false);
                    }
                },
                error: function() {
                    SEOAutoAdmin.showMessage($message, 'error', seoautoAdmin.strings.error);
                    $button.prop('disabled', false);
                }
            });
        },

        /**
         * Confirm delete
         */
        confirmDelete: function(e) {
            if (!confirm(seoautoAdmin.strings.confirmDelete)) {
                e.preventDefault();
            }
        },

        /**
         * Show message
         *
         * @param {jQuery} $element Element to show message in
         * @param {string} type     Message type (success, error, info)
         * @param {string} message  Message text
         */
        showMessage: function($element, type, message) {
            $element
                .removeClass('success error info')
                .addClass(type)
                .html(message)
                .slideDown();

            // Auto-hide success messages after 5 seconds
            if (type === 'success') {
                setTimeout(function() {
                    $element.slideUp();
                }, 5000);
            }
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        SEOAutoAdmin.init();
    });

})(jQuery);

/**
 * Settings Page - Modern Tabs & Interactions
 * Using vanilla JS for settings page specific functionality
 */
document.addEventListener('DOMContentLoaded', function() {
    'use strict';

    // Settings Tab switching
    var tabs = document.querySelectorAll('.seoauto-settings-tab');
    var panels = document.querySelectorAll('.seoauto-settings-panel');

    if (tabs.length > 0) {
        tabs.forEach(function(tab) {
            tab.addEventListener('click', function(e) {
                e.preventDefault();
                var targetTab = this.getAttribute('data-tab');

                // Remove active from all
                tabs.forEach(function(t) { t.classList.remove('active'); });
                panels.forEach(function(p) { p.classList.remove('active'); });

                // Add active to clicked
                this.classList.add('active');
                var targetPanel = document.getElementById('panel-' + targetTab);
                if (targetPanel) {
                    targetPanel.classList.add('active');
                }
            });
        });
    }

    // Toggle API key visibility (vanilla JS version for settings page)
    var toggleBtn = document.getElementById('toggle-key');
    var apiKeyInput = document.getElementById('seoauto_api_key');

    if (toggleBtn && apiKeyInput) {
        toggleBtn.addEventListener('click', function() {
            var icon = this.querySelector('.dashicons');
            if (apiKeyInput.type === 'password') {
                apiKeyInput.type = 'text';
                icon.classList.remove('dashicons-visibility');
                icon.classList.add('dashicons-hidden');
            } else {
                apiKeyInput.type = 'password';
                icon.classList.remove('dashicons-hidden');
                icon.classList.add('dashicons-visibility');
            }
        });
    }

    // Test connection button (vanilla JS version)
    var testBtn = document.getElementById('test-connection');
    var messageBox = document.getElementById('seoauto-message');

    if (testBtn && typeof seoautoAdmin !== 'undefined') {
        testBtn.addEventListener('click', function() {
            var apiKey = apiKeyInput ? apiKeyInput.value : '';
            if (!apiKey) {
                showMessage('Please enter an API key first.', 'error');
                return;
            }

            testBtn.disabled = true;
            testBtn.innerHTML = '<span class="dashicons dashicons-update-alt" style="animation: seoauto-spin 1s linear infinite;"></span> Testing...';

            var formData = new FormData();
            formData.append('action', 'seoauto_validate_api_key');
            formData.append('nonce', seoautoAdmin.nonce);
            formData.append('api_key', apiKey);

            fetch(seoautoAdmin.ajaxUrl, {
                method: 'POST',
                body: formData
            })
            .then(function(response) { return response.json(); })
            .then(function(data) {
                if (data.success) {
                    showMessage('Connection successful! ' + (data.data.message || ''), 'success');
                    // Reload after success to show updated connection status
                    setTimeout(function() { location.reload(); }, 1500);
                } else {
                    showMessage(data.data.message || 'Connection failed.', 'error');
                }
            })
            .catch(function() {
                showMessage('Connection error. Please try again.', 'error');
            })
            .finally(function() {
                testBtn.disabled = false;
                testBtn.innerHTML = '<span class="dashicons dashicons-yes-alt"></span> Test Connection';
            });
        });
    }

    // Reset plugin button (vanilla JS version)
    var resetBtn = document.getElementById('reset-plugin');

    if (resetBtn && typeof seoautoAdmin !== 'undefined') {
        resetBtn.addEventListener('click', function() {
            if (!confirm('Are you sure you want to reset the plugin? All settings will be lost.')) {
                return;
            }
            if (!confirm('This action cannot be undone. Are you really sure?')) {
                return;
            }

            resetBtn.disabled = true;
            resetBtn.innerHTML = '<span class="dashicons dashicons-update-alt" style="animation: seoauto-spin 1s linear infinite;"></span> Resetting...';

            var formData = new FormData();
            formData.append('action', 'seoauto_reset_plugin');
            formData.append('nonce', seoautoAdmin.nonce);

            fetch(seoautoAdmin.ajaxUrl, {
                method: 'POST',
                body: formData
            })
            .then(function(response) { return response.json(); })
            .then(function(data) {
                if (data.success) {
                    showMessage('Plugin reset successfully! Reloading...', 'success');
                    setTimeout(function() { location.reload(); }, 1500);
                } else {
                    showMessage(data.data.message || 'Reset failed.', 'error');
                    resetBtn.disabled = false;
                    resetBtn.innerHTML = '<span class="dashicons dashicons-trash"></span> Reset Plugin';
                }
            })
            .catch(function() {
                showMessage('Error resetting plugin.', 'error');
                resetBtn.disabled = false;
                resetBtn.innerHTML = '<span class="dashicons dashicons-trash"></span> Reset Plugin';
            });
        });
    }

    /**
     * Show message helper
     */
    function showMessage(text, type) {
        if (messageBox) {
            messageBox.textContent = text;
            messageBox.className = 'seoauto-inline-msg ' + type;
        }
    }
});
